package main

import (
	"fmt"
	"os"

	"github.com/spf13/pflag"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/indexer/internal/config"
	"a.yandex-team.ru/security/yadi/indexer/internal/indexer"
	"a.yandex-team.ru/security/yadi/indexer/internal/stat"
	"a.yandex-team.ru/security/yadi/libs/pypi"
)

var (
	opts struct {
		Target      string
		EnvType     string
		Concurrency int
		DryRun      bool
		Reindex     bool
		Verbose     bool
		ShowVersion bool
	}
)

func newConfig() (*config.Config, error) {
	switch opts.EnvType {
	case "dev":
		return &config.Config{
			DryRun:       opts.DryRun,
			Reindex:      opts.Reindex,
			Concurrency:  opts.Concurrency,
			YdbEndpoint:  "ydb-ru.yandex.net:2135",
			YdbDatabase:  "/ru/home/robot-yadi/mydb",
			YdbPath:      "/dev/pypi",
			YdbAuthToken: os.Getenv("YDB_TOKEN"),
		}, nil
	case "prod":
		return &config.Config{
			DryRun:       opts.DryRun,
			Reindex:      opts.Reindex,
			Concurrency:  opts.Concurrency,
			YdbEndpoint:  "ydb-ru.yandex.net:2135",
			YdbDatabase:  "/ru/home/robot-yadi/mydb",
			YdbPath:      "/prod/pypi",
			YdbAuthToken: os.Getenv("YDB_TOKEN"),
		}, nil
	default:
		return nil, xerrors.Errorf("unknown env type: %s", opts.EnvType)
	}
}

func handleError(msg string, err error) {
	if err == nil {
		return
	}

	_, _ = fmt.Fprintf(os.Stderr, "%s: %s\n", msg, err)
	os.Exit(1)
}

func main() {
	maxprocs.AdjustAuto()

	pflag.StringVar(&opts.Target, "target", "ext", "Target (ext/int/int-local)")
	pflag.StringVar(&opts.EnvType, "env-type", "dev", "Environment type (dev/prod)")
	pflag.IntVar(&opts.Concurrency, "concurrency", 8, "Concurrency level")
	pflag.BoolVar(&opts.DryRun, "dry-run", false, "Run in dry run mode")
	pflag.BoolVar(&opts.Reindex, "reindex", false, "Full reindex (skip any known versions and so on)")
	pflag.BoolVar(&opts.Verbose, "verbose", false, "Verbose output")
	pflag.BoolVar(&opts.ShowVersion, "version", false, "Show the current version")

	pflag.Parse()

	if opts.ShowVersion {
		fmt.Println(config.FullVersion())
		return
	}

	if opts.Verbose {
		simplelog.SetLevel(simplelog.DebugLevel)
	}

	cfg, err := newConfig()
	handleError("failed to create config", err)

	var targetPypi *pypi.PyPi
	var limits *stat.Limits
	switch opts.Target {
	case "ext":
		targetPypi, err = pypi.NewPyPiOrg()
		limits = &stat.Limits{
			ExpectedPkgs:       150000,
			AllowedPkgFails:    1000,
			AllowedPkgVerFails: 40000,
		}
	case "int":
		targetPypi, err = pypi.NewYandexTeamSimple()
		limits = &stat.Limits{
			ExpectedPkgs:       800,
			AllowedPkgFails:    50,
			AllowedPkgVerFails: 100,
		}
	case "int-local":
		targetPypi, err = pypi.NewYandexTeamLocal()
		limits = &stat.Limits{
			ExpectedPkgs:       800,
			AllowedPkgFails:    50,
			AllowedPkgVerFails: 100,
		}
	default:
		_, _ = fmt.Fprintf(os.Stderr, "unknown target: %s\n", opts.Target)
		os.Exit(1)
	}
	handleError("failed to create pypi", err)

	index, err := indexer.New(cfg)
	handleError("failed to create indexer", err)

	err = index.Start(targetPypi)
	if err != nil {
		_, _ = fmt.Fprintf(os.Stderr, "indexing failed: %s\n", err.Error())
		os.Exit(1)
	}

	stats := index.Stats()
	fmt.Println("done, stats:")
	fmt.Printf("\t- elapsed time: %s\n", stats.Elapsed)
	fmt.Printf("\t- total pkgs: %d\n", stats.Packages)
	fmt.Printf("\t- failed pkgs: %d\n", stats.PackageFails)
	fmt.Printf("\t- failed pkg versions: %d\n", stats.PackageVersionFails)
	fmt.Printf("\t- new pkg versions: %d\n", stats.NewPackageVersions)

	err = limits.CheckStats(stats)
	handleError("Ooops, indexing failed due to", err)
}
