package indexer

import (
	"context"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/indexer/internal/config"
	"a.yandex-team.ru/security/yadi/indexer/internal/db"
	"a.yandex-team.ru/security/yadi/indexer/internal/dbmodels"
	"a.yandex-team.ru/security/yadi/indexer/internal/priority"
	"a.yandex-team.ru/security/yadi/indexer/internal/stat"
	"a.yandex-team.ru/security/yadi/indexer/internal/versions"
	"a.yandex-team.ru/security/yadi/libs/pypi"
)

const (
	backoffTimeout = time.Second * 10
)

type (
	Indexer struct {
		dryRun      bool
		reindex     bool
		db          *db.DB
		concurrency int
		stats       *stat.Stat
	}
)

func New(cfg *config.Config) (*Indexer, error) {
	var database *db.DB
	database, err := db.New(context.Background(), cfg.DBConfig())
	if err != nil {
		return nil, xerrors.Errorf("failed to create database: %w", err)
	}

	return &Indexer{
		dryRun:      cfg.DryRun,
		reindex:     cfg.Reindex,
		db:          database,
		stats:       stat.New(),
		concurrency: cfg.Concurrency,
	}, nil
}

func (i *Indexer) Start(targetPyPi *pypi.PyPi) (resultErr error) {
	jobs := make(chan *pypi.Package, i.concurrency*2)
	wg := sync.WaitGroup{}
	wg.Add(i.concurrency)
	for k := 0; k < i.concurrency; k++ {
		go i.startIndexer(&wg, jobs, targetPyPi.Name())
	}

	for targetPyPi.Next() {
		pkg, err := targetPyPi.Package()
		if err != nil {
			simplelog.Error("failed to get package", "err", err)
			continue
		}

		jobs <- pkg
	}

	resultErr = targetPyPi.Error()

	close(jobs)
	wg.Wait()
	return
}

func (i *Indexer) Stats() stat.Stat {
	return i.stats.Stat()
}

func (i *Indexer) startIndexer(wg *sync.WaitGroup, jobs <-chan *pypi.Package, source string) {
	defer wg.Done()
	for pkg := range jobs {
		func(pkg *pypi.Package) {
			err := pkg.Resolve()
			if err != nil {
				simplelog.Error("failed to resolve pkg", "pkg_name", pkg.Name(), "err", err)
				return
			}
			i.stats.AddPackage()

			dbVersions := make(map[string]bool)
			dbPkg, err := i.db.LookupPackage(pkg.NormName())
			switch err {
			case nil:
				// ok, let's fill known versions if we on in reindex mode
				if i.reindex {
					break
				}

				for _, ver := range dbPkg.Versions {
					dbVersions[ver] = true
				}
			case db.ErrNotFound:
				// It's fine
			default:
				i.stats.AddPackageFail()
				simplelog.Error("failed to lookup pkg in DB",
					"pkg_name", pkg.Name(),
					"source", source,
					"err", err,
				)
				time.Sleep(backoffTimeout)
				return
			}

			switch {
			case dbPkg == nil:
				//ok
			case priority.PyPiCleanup(dbPkg.Source, source):
				if !i.dryRun {
					err = i.db.CleanUpPackage(dbPkg)
					if err != nil {
						simplelog.Error("failed to cleanup pkg", "pkg_name", pkg.Name(), "err", err)
						return
					}
				}

				simplelog.Warn("cleanup pkg", "pkg_name", pkg.Name())
				dbVersions = make(map[string]bool)
			case priority.PyPiSkip(dbPkg.Source, source):
				simplelog.Warn("skip pkg due to source priority", "pkg_name", pkg.Name())
				return
			}

			upstreamVersions := pkg.Versions()
			updData := &dbmodels.UpdatePackageData{
				Package: dbmodels.Package{
					Name:    pkg.NormName(),
					PkgName: pkg.Name(),
					Source:  source,
				},
				NewVersions: make([]dbmodels.LitePackageVersion, 0, len(upstreamVersions)),
			}

			if len(upstreamVersions) > 0 {
				for _, version := range upstreamVersions {
					if _, exists := dbVersions[version]; exists {
						continue
					}

					pkgVersion, err := pkg.Version(version)
					if err != nil {
						i.stats.AddPackageVersionFail()
						simplelog.Error("failed to parse pkg version",
							"pkg_name", pkg.Name(),
							"source", source,
							"err", err,
						)
						continue
					}

					requirements, err := pkgVersion.RequirementsJSON()
					if err != nil {
						simplelog.Error("failed to parse pkg requirements",
							"pkg_name", pkg.Name(),
							"source", source,
							"err", err,
						)
						continue
					}

					pkgVer := pkgVersion.Version()
					if pkgVer == "" {
						simplelog.Warn("empty pkg version - use index version",
							"pkg_name", pkg.Name(),
							"index_version", version,
						)
						// TODO(buglloc): ugly
						pkgVer = version
					}

					dbVersions[pkgVersion.Version()] = true
					updData.NewVersions = append(updData.NewVersions, dbmodels.LitePackageVersion{
						Version:      pkgVer,
						License:      pkgVersion.License(),
						PkgURL:       pkgVersion.DownloadURL(),
						Requirements: requirements,
					})
				}
			}

			if len(updData.NewVersions) > 0 {
				// If we have new versions - let's update versions
				updData.Package.Versions = make([]string, 0, len(dbVersions))
				for v := range dbVersions {
					updData.Package.Versions = append(updData.Package.Versions, v)
				}
				updData.Package.Versions = versions.SortVersions(updData.Package.Versions)
			} else if dbPkg != nil {
				// otherwise - try use previous
				updData.Package.Versions = dbPkg.Versions
			}

			if i.dryRun {
				newVersions := make([]string, len(updData.NewVersions))
				for k, v := range updData.NewVersions {
					i.stats.AddNewPackageVersion()
					newVersions[k] = v.Version
				}

				simplelog.Info("update pkg",
					"name", updData.Package.Name,
					"source", updData.Package.Source,
					"versions", updData.Package.Versions,
					"new_versions", newVersions,
				)
			} else {
				simplelog.Debug("update pkg",
					"name", updData.Package.Name,
					"new_versions", len(updData.NewVersions),
					"total_versions", len(dbVersions),
				)

				err := i.db.UpdatePackage(updData)
				if err != nil {
					i.stats.AddPackageFail()
					simplelog.Error("failed to update pkg", "pkg_name", pkg.Name(), "err", err)
				} else {
					i.stats.AddNewPackageVersions(uint64(len(updData.NewVersions)))
				}
			}
		}(pkg)
	}
}
