package cvs

import (
	"fmt"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
)

const (
	NoneSeverity     float32 = 0.0
	LowSeverity      float32 = 1.0
	MediumSeverity   float32 = 4.0
	HighSeverity     float32 = 7.0
	CriticalSeverity float32 = 9.0
	DefaultSeverity          = MediumSeverity
)

func RoundBySeverity(score float32) float32 {
	// See CVSSv3 for details: https://www.first.org/cvss/specification-document#i5
	switch {
	case score < LowSeverity:
		return NoneSeverity
	case score < MediumSeverity:
		return LowSeverity
	case score < HighSeverity:
		return MediumSeverity
	case score < CriticalSeverity:
		return HighSeverity
	default:
		return CriticalSeverity
	}
}

func ToSeverity(score float32) string {
	switch RoundBySeverity(score) {
	case NoneSeverity:
		return "None"
	case LowSeverity:
		return "Low"
	case MediumSeverity:
		return "Medium"
	case HighSeverity:
		return "High"
	case CriticalSeverity:
		fallthrough
	default:
		return "Critical"
	}
}

func FromSeverity(severity string) (float32, error) {
	switch strings.ToLower(severity) {
	case "info", "i":
		return NoneSeverity, nil
	case "low", "l":
		return LowSeverity, nil
	case "medium", "m":
		return MediumSeverity, nil
	case "high", "h":
		return HighSeverity, nil
	case "critical", "c":
		return CriticalSeverity, nil
	default:
		return NoneSeverity, xerrors.New("not valid severity")
	}
}

func FromLevel(level int) (score float32, err error) {
	switch level {
	case 0:
		score = DefaultSeverity
	case 1:
		score = LowSeverity
	case 2:
		score = MediumSeverity
	case 3:
		score = HighSeverity
	case 4:
		score = CriticalSeverity
	default:
		err = fmt.Errorf("unrecognized severity level (%d), must be from 1 to 4", level)
	}

	return
}
