package qualifiers

import (
	"strconv"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
)

// Qualifiers that we have successfully parsed:
// numeric version, known qualifiers (alpha, release, ...) or aliases (cr,...)
type KnownQualifier struct {
	orig    string
	Version int
}

// Known qualifiers and aliases
var qualifiersMap = map[string]int{
	"alpha": -5, "a": -5,
	"beta": -4, "b": -4,
	"milestone": -3, "m": -3,
	"rc": -2, "cr": -2,
	"snapshot": -1,
	"":         0, "ga": 0, "final": 0, "release": 0,
	"sp": +1,
}

func (q KnownQualifier) Compare(other Item) int {
	switch o := other.(type) {
	case KnownQualifier:
		if q.Version == o.Version {
			return 0
		}
		if q.Version < o.Version {
			return -1
		}
		return +1
	case UnknownQualifier:
		return +1 // 1-alpha-2 > 1-ad0263a1fe56651ca4a89fdd95c745ea24286498
	default:
		return +1
	}
}

func (q KnownQualifier) String() string {
	return q.orig
}

func NewQualifier(raw string) (qualifier KnownQualifier, err error) {
	qualifier.orig = raw
	raw = strings.ToLower(raw)
	if value, ok := qualifiersMap[raw]; ok {
		qualifier.Version = value
		return
	}
	value, err := strconv.ParseUint(raw, 10, 64)
	if err == nil {
		qualifier.Version = int(value)
		return
	}

	return qualifier, xerrors.Errorf("unknown qualifier: %s", raw)
}
