package qualifiers

import (
	"unicode"
)

const (
	digit = iota + 1
	letter
	separator
)

type (
	Item interface {
		Compare(other Item) int
		String() string
	}
	Qualifiers []Item
)

var separatorsTable = []*unicode.RangeTable{
	unicode.Dash,
}

func Parse(raw string) (result Qualifiers) {
	var runes [][]rune
	var kind, last int
	for _, r := range raw {
		switch true {
		case unicode.IsDigit(r):
			kind = digit
		case unicode.IsLetter(r):
			kind = letter
		case unicode.IsOneOf(separatorsTable, r):
			last = separator
			continue
		default:
			kind = letter
		}

		if kind == last {
			runes[len(runes)-1] = append(runes[len(runes)-1], r)
		} else {
			runes = append(runes, []rune{r})
		}
		last = kind
	}
	var err error
	var q Item
	for _, s := range runes {
		if len(s) > 0 {
			q, err = NewQualifier(string(s))
			if err != nil {
				q = NewUnknown(string(s))
			}
			result = append(result, q)
		}
	}
	return result
}

// comparing of qualifiers
func (q Qualifiers) Compare(other Qualifiers) int {
	if len(q) == 0 && len(other) == 0 {
		return 0
	}

	if len(q) < len(other) {
		q = append(q, zerosPadding(len(other)-len(q))...)
	}

	if len(other) < len(q) {
		other = append(other, zerosPadding(len(q)-len(other))...)
	}

	var i, result int
	for i < len(q) {
		result = q[i].Compare(other[i])
		if result != 0 {
			return result
		}
		i++
	}
	return result
}

func zerosPadding(n int) (r []Item) {
	for i := 0; i < n; i++ {
		r = append(r, KnownQualifier{Version: 0})
	}
	return
}
