package maven

import (
	"fmt"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
)

type border rune

const (
	leftOpenBorder   border = '('
	leftCloseBorder  border = '['
	rightOpenBorder  border = ')'
	rightCloseBorder border = ']'
	orSeparator             = ','
)

type comparator func(Version, Version) bool

var (
	compEQ comparator = func(v1 Version, v2 Version) bool {
		return v1.Compare(v2) == 0
	}
	compGT = func(v1 Version, v2 Version) bool {
		return v1.Compare(v2) == 1
	}
	compGE = func(v1 Version, v2 Version) bool {
		return v1.Compare(v2) >= 0
	}
	compLT = func(v1 Version, v2 Version) bool {
		return v1.Compare(v2) == -1
	}
	compLE = func(v1 Version, v2 Version) bool {
		return v1.Compare(v2) <= 0
	}
)

type versionRange struct {
	v Version
	c comparator
}

// rangeFunc creates a Range from the given versionRange.
func (vr *versionRange) rangeFunc() Range {
	return Range(func(v Version) bool {
		return vr.c(v, vr.v)
	})
}

// Range represents a range of versions.
// A Range can be used to check if a Version satisfies it:
type Range func(Version) bool

// OR combines the existing Range with another Range using logical OR.
func (rf Range) OR(f Range) Range {
	return Range(func(v Version) bool {
		return rf(v) || f(v)
	})
}

// AND combines the existing Range with another Range using logical AND.
func (rf Range) AND(f Range) Range {
	return Range(func(v Version) bool {
		return rf(v) && f(v)
	})
}

// ParseRange parses a range in Maven manner and returns a Range
func ParseRange(s string) (Range, error) {
	//parts := SplitAndTrim(s)
	orParts, err := splitORParts(s)
	if err != nil {
		return nil, err
	}

	var orFn Range
	for _, p := range orParts {
		var andFn Range
		for _, ap := range p {
			opStr, vStr, err := splitBorderVersion(ap)
			if err != nil {
				return nil, err
			}
			vr, err := buildVersionRange(opStr, vStr)
			if err != nil {
				return nil, fmt.Errorf("could not parse Range %q: %s", ap, err)
			}
			rf := vr.rangeFunc()

			// Set function
			if andFn == nil {
				andFn = rf
			} else { // Combine with existing function
				andFn = andFn.AND(rf)
			}
		}
		if orFn == nil {
			orFn = andFn
		} else {
			orFn = orFn.OR(andFn)
		}

	}
	return orFn, nil
}

// splitORParts splits string by ','
func splitORParts(str string) ([][]string, error) {
	var ORparts [][]string

	var orIndexes = []int{-1}
	for _, separatorIndex := range allIndexesRune(str, orSeparator) {
		if outsideBorders(str, separatorIndex) {
			orIndexes = append(orIndexes, separatorIndex)
		}
	}
	orIndexes = append(orIndexes, len(str)) // to split string by indexes correctly

	for i := 0; i < len(orIndexes)-1; i++ {
		part := str[orIndexes[i]+1 : orIndexes[i+1]]

		if part == "" {
			return nil, xerrors.Errorf("empty subversion in: %s", str)
		}

		// versions inside borders are separated by "," too
		subparts := strings.Split(part, ",")
		if len(subparts) > 2 {
			return nil, xerrors.Errorf("wrong number of separators in version: %s", str)
		}

		for i := range subparts {
			subparts[i] = strings.TrimSpace(subparts[i])
			if subparts[i] == "" {
				return nil, xerrors.Errorf("empty version in: %s", str)
			}
		}

		ORparts = append(ORparts, subparts)
	}

	return ORparts, nil
}

// outsideBorders checks that rune on position is located outside borders (brackets)
// outsideBorders("[1,2)", 2) => false (the 2-nd rune ',' is inside brackets)
// outsideBorders("[1,2), [5]", 5) => true (the 5-th rune ',' is out of brackets)
func outsideBorders(str string, position int) (outside bool) {
	s := []rune(str)

	if position <= 0 || position >= (len(s)-1) {
		return false
	}

	// check left part of substring
	for i := position; i > 0; i-- {
		if isBorder(leftCloseBorder, leftOpenBorder)(s[i]) {
			return false
		}
		if isBorder(rightCloseBorder, rightOpenBorder)(s[i]) {
			outside = true
			break
		}
	}

	// check right part of substring
	for i := position; i < len(s); i++ {
		if isBorder(rightCloseBorder, rightOpenBorder)(s[i]) {
			return false
		}
		if isBorder(leftCloseBorder, leftOpenBorder)(s[i]) {
			outside = outside && true
			break
		}
	}

	return
}

func allIndexesRune(s string, r rune) (result []int) {
	var substring = []rune(s)
	for index := range substring {
		if substring[index] == r {
			result = append(result, index)
		}
	}
	return
}

// buildVersionRange takes a slice of 2: operator and version
// and builds a versionRange, otherwise an error.
func buildVersionRange(opStr, vStr string) (*versionRange, error) {
	var v Version
	c := parseComparator(opStr)
	if c == nil {
		return nil, fmt.Errorf("could not parse comparator %q in %q", opStr, strings.Join([]string{opStr, vStr}, ""))
	}

	opRune := []rune(opStr)[0]
	if isBorder(rightOpenBorder, rightCloseBorder)(opRune) && vStr == "" {
		// ,] or ,) cases -> right version is infinite
		return &versionRange{
			v: InfiniteVersion,
			c: c,
		}, nil
	}
	v, err := ParseVersion(vStr)
	if err != nil {
		return nil, fmt.Errorf("could not parse version %q in %q: %s", vStr, strings.Join([]string{opStr, vStr}, ""), err)
	}

	return &versionRange{
		v: v,
		c: c,
	}, nil

}

// returns true if the rune is one of the borders
func isBorder(borders ...border) func(r rune) bool {
	return func(r rune) bool {
		for _, b := range borders {
			if r == rune(b) {
				return true
			}
		}
		return false
	}
}

// splitBorderVersion splits the border from the version.
// Input must be free of leading or trailing spaces.
func splitBorderVersion(s string) (string, string, error) {
	leftIndex := strings.IndexFunc(s, isBorder(leftOpenBorder, leftCloseBorder))
	rightIndex := strings.IndexFunc(s, isBorder(rightOpenBorder, rightCloseBorder))

	switch {
	case leftIndex >= 0 && rightIndex >= 0:
		// [1.2.3] or (1.2.3) or (1.2.3] or [1.2.3)
		if r := []rune(s); isBorder(leftCloseBorder)(r[0]) && isBorder(rightCloseBorder)(r[len(r)-1]) {
			// only [1.2.3] is valid
			return "=", string(r[1 : len(r)-1]), nil
		}
		return "", "", fmt.Errorf("invalid borders in version: %q", s)
	case leftIndex >= 0 && rightIndex < 0:
		// (1.2.3 or [1.2.3
		return strings.TrimSpace(s[:leftIndex+1]), s[leftIndex+1:], nil
	case rightIndex >= 0 && leftIndex < 0:
		// 1.2.3)
		return strings.TrimSpace(s[rightIndex:]), s[0:rightIndex], nil
	}
	return "", "", fmt.Errorf("could not get border from string: %q", s)
}

func parseComparator(s string) comparator {
	switch s {
	case "=":
		return compEQ
	case "(":
		return compGT
	case "[":
		return compGE
	case ")":
		return compLT
	case "]":
		return compLE
	}
	return nil
}
