package maven

import (
	"strconv"
	"strings"

	"github.com/blang/semver/v4"

	"a.yandex-team.ru/security/yadi/libs/maven/qualifiers"
	"a.yandex-team.ru/security/yadi/libs/maven/utils"
)

const numbers = "0123456789"

type versionType int

const (
	normal versionType = iota
	zero
	infinite
)

type Version struct {
	semver.Version
	vtype      versionType
	Qualifiers qualifiers.Qualifiers
}

var (
	ZeroVersion     = Version{vtype: zero}
	InfiniteVersion = Version{vtype: infinite}
)

// Parse parses version string in Maven manner and returns a validated Version or error
// (this function is the fork of semver.ParseTolerant)
func ParseVersion(s string) (Version, error) {
	s = strings.TrimSpace(s)
	s = strings.TrimPrefix(s, "v")

	if s == "" {
		return ZeroVersion, nil
	}

	// Split into major.minor.(patch+pr+meta)
	parts := strings.SplitN(s, ".", 3)

	// Get a build number and qualifier
	var tail string
	if strings.ContainsAny(parts[len(parts)-1], "-.") {
		subparts := utils.SplitNWithAnySep(parts[len(parts)-1], ".-", 2)
		tail = "-" + subparts[len(subparts)-1]
		if len(subparts) > 1 {
			parts = append(parts[:len(parts)-1], subparts[0])
		}
	}

	// Fill up shortened versions and move qualifiers to the tail, eg
	// 1.2.alpha -> 1.2.0-alpha
	// RELEASE -> 0.0.0-RELEASE
	// 1alpha -> 0.0.0-1alpha
	var i int
	for {
		if !containsOnly(parts[i], numbers) {
			tail = "-" + parts[i] + tail
			parts[i] = "0"
		} else if len(parts) < 3 {
			parts = append(parts, "0")
			i++
		} else if i < 2 {
			i++
		} else {
			break
		}
	}

	// Remove leading zeros from parts
	for i, p := range parts {
		if len(p) > 1 {
			p = strings.TrimLeft(p, "0")
			if len(p) == 0 {
				p = "0" + p
			}
			parts[i] = p
		}
	}

	s = strings.Join(parts, ".")
	v, err := semver.Parse(s)

	q := qualifiers.Parse(tail)

	return Version{
		Version:    v,
		Qualifiers: q,
		vtype:      normal,
	}, err
}

func (v Version) Compare(o Version) int {
	switch {
	case v.vtype == infinite && o.vtype == infinite:
		return 0
	case v.vtype == infinite:
		return +1
	case o.vtype == infinite:
		return -1
	}

	if v.Major != o.Major {
		if v.Major > o.Major {
			return +1
		}
		return -1
	}

	if v.Minor != o.Minor {
		if v.Minor > o.Minor {
			return +1
		}
		return -1
	}

	if v.Patch != o.Patch {
		if v.Patch > o.Patch {
			return +1
		}
		return -1
	}

	// otherwise compare qualifiers
	return v.Qualifiers.Compare(o.Qualifiers)
}

func (v Version) String() string {
	b := make([]byte, 0, 5)
	b = strconv.AppendUint(b, v.Major, 10)
	b = append(b, '.')
	b = strconv.AppendUint(b, v.Minor, 10)
	b = append(b, '.')
	b = strconv.AppendUint(b, v.Patch, 10)

	if len(v.Qualifiers) > 0 {
		b = append(b, '-')
		b = append(b, v.Qualifiers[0].String()...)

		for _, q := range v.Qualifiers[1:] {
			b = append(b, '.')
			b = append(b, q.String()...)
		}
	}

	if len(v.Build) > 0 {
		b = append(b, '+')
		b = append(b, v.Build[0]...)

		for _, build := range v.Build[1:] {
			b = append(b, '.')
			b = append(b, build...)
		}
	}

	return string(b)
}

func containsOnly(s string, set string) bool {
	return strings.IndexFunc(s, func(r rune) bool {
		return !strings.ContainsRune(set, r)
	}) == -1
}
