package policy

import (
	"bufio"
	"fmt"
	"io"
	"sort"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
)

type (
	recordList []string
	Policy     struct {
		contribPath string
		comments    []string
		allowRules  recordList
		denyRules   recordList
	}
)

const (
	policyAllowDirective  = "ALLOW"
	policyDenyDirective   = "DENY"
	policySrcAny          = ".*"
	policyRecordDelimiter = "->"
	policyPartsCount      = 4
)

func (rl recordList) Len() int      { return len(rl) }
func (rl recordList) Swap(i, j int) { rl[i], rl[j] = rl[j], rl[i] }
func (rl recordList) Less(i, j int) bool {
	return rl[i] < rl[j]
}

func NewPolicy(dst string) Policy {
	return Policy{
		contribPath: dst,
	}
}

func Read(reader io.Reader) (map[string]*Policy, error) {
	result := make(map[string]*Policy)
	scan := bufio.NewScanner(reader)

	comments := make([]string, 0)
	for scan.Scan() {
		line := strings.TrimSpace(scan.Text())
		if line == "" {
			// flush comments and go on
			comments = nil
			continue
		}

		if idx := strings.Index(line, "#"); idx >= 0 {
			comments = append(comments, strings.TrimSpace(line[idx+1:]))
			continue
		}

		parts := strings.Fields(line)
		if len(parts) != policyPartsCount {
			return nil, xerrors.Errorf("invalid policy record: %s", line)
		}

		if parts[0] != policyAllowDirective &&
			parts[0] != policyDenyDirective {
			return nil, fmt.Errorf("policy should be allowing or disallowing, not: %s", parts[0])
		}

		if parts[2] != policyRecordDelimiter {
			return nil, fmt.Errorf("bad record delimiter: %s", parts[2])
		}

		contrib := parts[3]
		target, ok := result[contrib]
		if !ok {
			target = new(Policy)
			result[contrib] = target
		}
		target.contribPath = contrib

		switch parts[0] {
		case policyAllowDirective:
			target.allowRules = append(target.allowRules, parts[1])
		case policyDenyDirective:
			target.denyRules = append(target.denyRules, parts[1])
		}

		target.comments = append(target.comments, comments...)
		comments = nil
	}

	if err := scan.Err(); err != nil {
		return nil, err
	}

	return result, nil
}
func (p *Policy) DenyAll() *Policy {
	p.denyRules = recordList{policySrcAny}
	return p
}

func (p *Policy) Deny(src string) *Policy {
	p.denyRules = append(recordList{src}, p.denyRules...)
	return p
}

func (p *Policy) Allow(src string) *Policy {
	p.allowRules = append(p.allowRules, src)
	return p
}

func (p *Policy) Comment(text string) *Policy {
	if strings.ContainsAny(text, "\r\n") {
		panic("multiline comments are not supported")
	}
	p.comments = append(p.comments, text)
	return p
}

func (p *Policy) Records() (out []string) {
	// first we write comments
	for _, comment := range p.comments {
		out = append(out, fmt.Sprintf("# %s", comment))
	}

	// then we write allow rules
	sort.Sort(p.allowRules)
	for _, src := range p.allowRules {
		out = append(out, fmt.Sprintf("%s %s %s %s", policyAllowDirective, src, policyRecordDelimiter, p.contribPath))
	}

	// then we write deny rules
	sort.Sort(p.denyRules)
	for _, src := range p.denyRules {
		out = append(out, fmt.Sprintf("%s %s %s %s", policyDenyDirective, src, policyRecordDelimiter, p.contribPath))
	}

	return
}

func (p *Policy) String() string {
	out := strings.Builder{}
	out.WriteString("\n")
	if records := p.Records(); len(records) > 0 {
		out.WriteString(strings.Join(records, "\n"))
		out.WriteString("\n")
	}
	return out.String()
}
