package pkgparser

import (
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/archer"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const (
	eggFilePkgInfo int = iota
	eggFileRequires
)

var (
	eggWalkOpts = archer.FileWalkOpts{
		Once: true,
		Patterns: []archer.WalkPattern{
			{
				ID:      eggFilePkgInfo,
				Marker:  "PKG-INFO",
				Pattern: "EGG-INFO/PKG-INFO",
			},
			{
				ID:      eggFileRequires,
				Marker:  "requires.txt",
				Pattern: "EGG-INFO/requires.txt",
			},
		},
	}
)

func ParseEggPackage(pkgPath string, archWalker archer.Walker) (pkg *PkgInfo, err error) {
	pkg = new(PkgInfo)

	pkgInfoParsed := false
	pkgReqParsed := false
	err = archWalker.FileWalk(
		pkgPath,
		eggWalkOpts,
		func(targetPath string, fileId int, reader archer.SizeReader) error {
			switch fileId {
			case srcFilePkgInfo:
				if pkgInfoParsed {
					simplelog.Warn("multiple PKG-INFO files")
					break
				}

				pkgInfoParsed = true
				err = ParseSrcPkgInfo(reader, pkg)
			case srcFileRequires:
				if pkgReqParsed {
					simplelog.Warn("multiple requires.txt files")
					break
				}

				pkgReqParsed = true
				pkg.Requires, pkg.Extras, err = ParseSrcRequires(reader)
			}

			if pkgInfoParsed && pkgReqParsed {
				return archer.ErrStop
			}
			return nil
		},
	)

	if err != nil {
		err = xerrors.Errorf("failed to parse egg pkg: %w", err)
	}

	if pkg == nil || pkg.Name == "" {
		err = xerrors.Errorf("failed to parse egg pkg: no suitable files found")
	}
	return
}
