package pkgparser

import (
	"bufio"
	"io"
	"regexp"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/archer"
	"a.yandex-team.ru/security/libs/go/simplelog"
)

const (
	srcFilePkgInfo int = iota
	srcFileRequires
	srcFileRequirements
)

var (
	srcRequirementRe = regexp.MustCompile(`^(?P<name>[\w.\-]+)\s*(?:\[(?P<extras>[\w.\-,]+)])?\s*(?P<ver>.*)\s*$`)
	srcWalkOpts      = archer.FileWalkOpts{
		Once: true,
		Patterns: []archer.WalkPattern{
			{
				ID:      srcFilePkgInfo,
				Marker:  "PKG-INFO",
				Pattern: "*/PKG-INFO",
			},
			{
				ID:      srcFileRequires,
				Marker:  "requires.txt",
				Pattern: "*/*.egg-info/requires.txt",
			},
			//
			//{
			//	ID:      srcFileRequirements,
			//	Marker:  "requirements.txt",
			//	Pattern: "*/requirements.txt",
			//},
		},
	}
)

func ParseSrcPackage(pkgPath string, archWalker archer.Walker) (pkg *PkgInfo, err error) {
	pkg = new(PkgInfo)

	pkgInfoParsed := false
	pkgReqParsed := false
	err = archWalker.FileWalk(
		pkgPath,
		srcWalkOpts,
		func(targetPath string, fileId int, reader archer.SizeReader) error {
			switch fileId {
			case srcFilePkgInfo:
				if pkgInfoParsed {
					simplelog.Warn("multiple PKG-INFO files")
					break
				}

				pkgInfoParsed = true
				err = ParseSrcPkgInfo(reader, pkg)
			case srcFileRequires, srcFileRequirements:
				if pkgReqParsed {
					simplelog.Warn("multiple requirements files")
					break
				}

				pkgReqParsed = true
				pkg.Requires, pkg.Extras, err = ParseSrcRequires(reader)
			}

			if pkgInfoParsed && pkgReqParsed {
				return archer.ErrStop
			}
			return nil
		},
	)

	if err != nil {
		err = xerrors.Errorf("failed to parse src pkg: %w", err)
	}

	if pkg == nil || pkg.Name == "" {
		err = xerrors.Errorf("failed to parse src pkg: no suitable files found")
	}
	return
}

func ParseSrcRequires(reader io.Reader) (requires []Require, extras map[string][]Require, err error) {
	// Werkzeug>=0.14
	// [extra]
	// pytest>=3
	// tox

	requires = make([]Require, 0)
	extra := ""
	scanner := bufio.NewScanner(reader)
	for scanner.Scan() {
		line := strings.TrimSpace(scanner.Text())
		if line == "" {
			continue
		}

		switch line[0] {
		case '@':
			// skip URI
			continue
		case '#':
			// skip comments
			continue
		case '-':
			// unsupported and must not be included in packages
			continue
		case '[':
			extra = strings.TrimSpace(line[1 : len(line)-1])
			continue
		}

		parsed := srcRequirementRe.FindStringSubmatch(line)
		if parsed == nil {
			err = xerrors.Errorf("failed to parse requirement: %s", line)
			return
		}

		req := Require{
			Name:     strings.TrimSpace(parsed[1]),
			Versions: strings.TrimSpace(parsed[3]),
		}

		requiredExtras := strings.TrimSpace(parsed[2])
		if requiredExtras != "" {
			req.Extras = strings.Split(requiredExtras, ",")
		}

		if extra != "" {
			if extras == nil {
				extras = make(map[string][]Require)
			}

			extras[extra] = append(extras[extra], req)
		} else {
			requires = append(requires, req)
		}
	}
	return
}

func ParseSrcPkgInfo(reader io.Reader, pkg *PkgInfo) (err error) {
	scanner := bufio.NewScanner(reader)
	for scanner.Scan() {
		line := scanner.Text()
		// In src pkg some field (e.g. Classifiers) placed _after_ body
		//if line == "" {
		//	// end of headers
		//	break
		//}

		switch {
		case strings.HasPrefix(line, "Name:"):
			pkg.Name = strings.TrimSpace(line[5:])
		case strings.HasPrefix(line, "Version:"):
			pkg.Version = strings.TrimSpace(line[8:])
		case strings.HasPrefix(line, "License:") && pkg.License == "":
			pkg.License = strings.TrimSpace(line[8:])
		case strings.HasPrefix(line, "Classifier: License"):
			// parse license from trove classifiers
			// must override "raw" license field
			parts := strings.Split(line[19:], "::")
			if len(parts) < 3 {
				break
			}

			pkg.License = strings.TrimSpace(parts[len(parts)-1])
		}
	}
	return
}
