package pkgparser

import (
	"bufio"
	"io"
	"regexp"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/archer"
)

var (
	wheelRequirementRe = regexp.MustCompile(`^\s*(?P<name>[\w.\-]+)\s*(?:\[(?P<extras>[\w.\-,]+)])?\s*(?:\(?(?P<ver>[^()]*)\)?)\s*$`)
	wheelWalkOpts      = archer.FileWalkOpts{
		Once: true,
		Patterns: []archer.WalkPattern{
			{
				ID:      0,
				Marker:  "METADATA",
				Pattern: "*.dist-info/METADATA",
			},
		},
	}
)

func ParseWheelPackage(pkgPath string, archWalker archer.Walker) (pkg *PkgInfo, err error) {
	err = archWalker.FileWalk(
		pkgPath,
		wheelWalkOpts,
		func(targetPath string, fileId int, reader archer.SizeReader) error {
			pkg, err = ParseWheelMetadata(reader)
			return archer.ErrStop
		},
	)

	if err != nil {
		err = xerrors.Errorf("failed to parse wheel pkg: %w", err)
	}

	if pkg == nil || pkg.Name == "" {
		err = xerrors.Errorf("failed to parse wheel: no suitable files found")
	}
	return
}

func ParseWheelMetadata(reader io.Reader) (pkg *PkgInfo, err error) {
	pkg = new(PkgInfo)
	scanner := bufio.NewScanner(reader)
	for scanner.Scan() {
		line := scanner.Text()
		if line == "" {
			// end of headers
			break
		}

		switch {
		case strings.HasPrefix(line, "Name:"):
			pkg.Name = strings.TrimSpace(line[5:])
		case strings.HasPrefix(line, "Version:"):
			pkg.Version = strings.TrimSpace(line[8:])
		case strings.HasPrefix(line, "License:") && pkg.License == "":
			pkg.License = strings.TrimSpace(line[8:])
		case strings.HasPrefix(line, "Classifier: License"):
			// parse license from trove classifiers
			// must override "raw" license field
			parts := strings.Split(line[19:], "::")
			if len(parts) < 3 {
				break
			}

			pkg.License = strings.TrimSpace(parts[len(parts)-1])
		case strings.HasPrefix(line, "Requires-Dist:") && len(line) > 15:
			// Metadata 2.1:
			// Requires-Dist: argon2-cffi (>=16.1.0) ; extra == 'argon2'
			// Requires-Dist: pyparsing (>=1.5.5)
			// Requires-Dist: argparse (>=1.4.0); python_version < "3.2"
			// Requires-Dist: requests[security] (>=1.0.0)
			// Metadata 2.0:
			// Requires-Dist: PySocks>=1.5.6,<2.0,!=1.5.7; extra == 'socks'

			values := strings.Split(line[14:], ";")
			parsed := wheelRequirementRe.FindStringSubmatch(values[0])
			if parsed == nil {
				err = xerrors.Errorf("failed to parse requirement: %s", values[0])
				return
			}

			req := Require{
				Name:     strings.TrimSpace(parsed[1]),
				Versions: strings.TrimSpace(parsed[3]),
			}

			requiredExtras := strings.TrimSpace(parsed[2])
			if requiredExtras != "" {
				req.Extras = strings.Split(requiredExtras, ",")
			}

			isExtra := false
			for _, val := range values[1:] {
				if strings.HasPrefix(val, " extra == '") {
					isExtra = true
					extraName := strings.TrimSpace(val[11:])
					extraName = extraName[:len(extraName)-1]

					if pkg.Extras == nil {
						// TODO(buglloc): ?!!!!
						pkg.Extras = make(map[string][]Require)
					}
					pkg.Extras[extraName] = append(pkg.Extras[extraName], req)
					break
				}
			}

			if !isExtra {
				pkg.Requires = append(pkg.Requires, req)
			}
		}
	}
	return
}
