package versionarium

// Implementation of versioning comparison for C++ libraries
// Strategy:
// By default we try to compare versions using semver library
// Otherwise we fallback to simple string comparison

import (
	"encoding/json"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/semver"
)

const (
	cppVersionType = "cppVersion"
)

var (
	_ Version      = (*cppVersion)(nil)
	_ VersionRange = (*cppVersionRange)(nil)
)

type (
	cppVersion struct {
		semver *semver.Version
		verStr string
	}

	cppVersionRange struct {
		rawConstraints string
		versionUpTo    string
		boundIncluded  bool
		constraints    *semver.Constraints
	}
)

func newCppVersion(rawVersion string) (cppVersion, error) {
	ver, err := semver.NewVersion(rawVersion)
	if err != nil {
		// Ok, fallback. Lets work with that string
		return cppVersion{
			semver: nil,
			verStr: Canonize(rawVersion),
		}, nil
	}
	return cppVersion{
		semver: ver,
		verStr: Canonize(rawVersion),
	}, nil
}

func parseRawVersion(rawVersionRange string) (string, bool, error) {
	// rawVersionRange format example "<=1.2.3"
	switch {
	case strings.HasPrefix(rawVersionRange, "<="):
		return strings.TrimPrefix(Canonize(rawVersionRange), "<="), true, nil
	case strings.HasPrefix(rawVersionRange, "<"):
		return strings.TrimPrefix(Canonize(rawVersionRange), "<"), false, nil
	case rawVersionRange == "":
		// No version data, that is ok
		return "", false, nil
	default:
		return "", false, xerrors.Errorf("Invalid raw version range format: %s", rawVersionRange)
	}
}

func backToRawVersion(versionUpTo string, boundIncluded bool) string {
	if boundIncluded {
		return "<=" + versionUpTo
	}
	return "<" + versionUpTo
}

func newCppVersionRange(rawVersionRange string) (cppVersionRange, error) {
	versionUpTo, boundIncluded, err := parseRawVersion(rawVersionRange)
	return cppVersionRange{
		versionUpTo:   versionUpTo,
		boundIncluded: boundIncluded,
	}, err
}

func (c cppVersionRange) Type() string {
	return cppVersionType
}

func (c cppVersionRange) String() string {
	return backToRawVersion(c.versionUpTo, c.boundIncluded)
}

func (c cppVersionRange) Check(v Version) bool {
	if v.String() == "" {
		return false // Ignore packages with empty version string
	}
	thisVersion, _ := newCppVersion(c.versionUpTo)
	cmpVal := v.Compare(thisVersion)
	if c.boundIncluded {
		return cmpVal <= 0
	}
	return cmpVal < 0
}

func (c cppVersion) MarshalJSON() ([]byte, error) {
	return json.Marshal(c.verStr)
}

func (c cppVersion) Type() string {
	return cppVersionType
}

func (c cppVersion) String() string {
	return c.verStr
}

func (c cppVersion) ReleaseInfo() string {
	if c.semver != nil {
		return c.semver.Prerelease()
	}
	return ""
}

func (c cppVersion) BuildInfo() string {
	if c.semver != nil {
		return c.semver.Metadata()
	}
	return ""
}

func (c cppVersion) Compare(o Version) int {
	ver, ok := o.(cppVersion)
	if !ok {
		panic(xerrors.Errorf(errWrongType, c.Type(), ver.Type()))
	}
	if c.semver != nil && ver.semver != nil {
		return c.semver.Compare(ver.semver)
	}
	// Fallback here
	return strings.Compare(c.verStr, ver.verStr)
}

func (c cppVersion) LessThan(o Version) bool {
	return c.Compare(o) < 0
}

func (c cppVersion) GreaterThan(o Version) bool {
	return c.Compare(o) > 0
}

func (c cppVersion) Equal(o Version) bool {
	return c.Compare(o) == 0
}

// Trim spaces and to lowercase
// Important for version comparison
func Canonize(smth string) string {
	return strings.TrimSpace(strings.ToLower(smth))
}
