package versionarium

import (
	"encoding/json"
	"fmt"
	"strings"

	"github.com/blang/semver/v4"
)

const (
	goSemverType = "gosemver"
)

var (
	_ Version      = (*goVersion)(nil)
	_ VersionRange = (*goVersionRange)(nil)
)

type (
	goVersion struct {
		ver    semver.Version
		verStr string
	}

	goVersionRange struct {
		rawConstraints string
		constraints    semver.Range
	}
)

func newGoSemverVersion(rawVersion string) (goVersion, error) {
	ver, err := semver.ParseTolerant(rawVersion)
	if err != nil {
		return goVersion{}, err
	}

	return goVersion{
		ver:    ver,
		verStr: ver.String(),
	}, nil
}

func newGoSemverRange(rawVersionRange string) (goVersionRange, error) {
	constraint, err := semver.ParseRange(rawVersionRange)
	if err != nil {
		return goVersionRange{}, err
	}
	return goVersionRange{
		constraints:    constraint,
		rawConstraints: rawVersionRange,
	}, nil
}

func (v goVersion) Type() string {
	return goSemverType
}

func (v goVersion) String() string {
	return v.verStr
}

func (v goVersion) MarshalJSON() ([]byte, error) {
	return json.Marshal(v.verStr)
}

func (v goVersion) ReleaseInfo() string {
	out := make([]string, len(v.ver.Pre))
	for i, p := range v.ver.Pre {
		out[i] = p.String()
	}
	return strings.Join(out, ".")
}

func (v goVersion) BuildInfo() string {
	return strings.Join(v.ver.Build, ".")
}

func (v goVersion) Compare(o Version) int {
	ver, ok := o.(goVersion)
	if !ok {
		panic(fmt.Sprintf(errWrongType, v.Type(), ver.Type()))
	}
	return v.ver.Compare(ver.ver)
}

func (v goVersion) LessThan(o Version) bool {
	return v.Compare(o) < 0
}

func (v goVersion) GreaterThan(o Version) bool {
	return v.Compare(o) > 0
}

func (v goVersion) Equal(o Version) bool {
	return v.Compare(o) == 0
}

func (r goVersionRange) Type() string {
	return goSemverType
}

func (r goVersionRange) String() string {
	return r.rawConstraints
}

func (r goVersionRange) Check(v Version) bool {
	ver, ok := v.(goVersion)
	if !ok {
		panic(fmt.Sprintf(errWrongType, r.Type(), v.Type()))
	}

	return r.constraints(ver.ver)
}
