package versionarium

import (
	"encoding/json"
	"fmt"
	"strings"

	"a.yandex-team.ru/security/yadi/libs/maven"
)

const mavenType = "maven"

var (
	_ Version      = (*mavenVersion)(nil)
	_ VersionRange = (*mavenRange)(nil)
)

type (
	mavenVersion struct {
		mvnVersion maven.Version
		verStr     string
	}

	mavenRange struct {
		rawRange string
		mvnRange maven.Range
	}
)

func newMavenVersion(rawVersion string) (mavenVersion, error) {
	ver, err := maven.ParseVersion(rawVersion)
	if err != nil {
		return mavenVersion{}, err
	}

	return mavenVersion{
		mvnVersion: ver,
		verStr:     ver.String(),
	}, nil
}

func newMavenRange(rawVersionRange string) (mavenRange, error) {
	rng, err := maven.ParseRange(rawVersionRange)
	if err != nil {
		return mavenRange{}, err
	}

	return mavenRange{
		rawRange: rawVersionRange,
		mvnRange: rng,
	}, nil
}

func (v mavenVersion) Type() string {
	return mavenType
}

func (v mavenVersion) String() string {
	return v.verStr
}

func (v mavenVersion) MarshalJSON() ([]byte, error) {
	return json.Marshal(v.verStr)
}

func (v mavenVersion) ReleaseInfo() string {
	out := make([]string, len(v.mvnVersion.Qualifiers))
	for i, q := range v.mvnVersion.Qualifiers {
		out[i] = q.String()
	}
	return strings.Join(out, "-")
}

func (v mavenVersion) BuildInfo() string {
	return strings.Join(v.mvnVersion.Build, ".")
}

func (v mavenVersion) Compare(o Version) int {
	ver, ok := o.(mavenVersion)
	if !ok {
		panic(fmt.Sprintf(errWrongType, v.Type(), ver.Type()))
	}
	return v.mvnVersion.Compare(ver.mvnVersion)
}

func (v mavenVersion) LessThan(o Version) bool {
	return v.Compare(o) < 0
}

func (v mavenVersion) GreaterThan(o Version) bool {
	return v.Compare(o) > 0
}

func (v mavenVersion) Equal(o Version) bool {
	return v.Compare(o) == 0
}

func (r mavenRange) Type() string {
	return mavenType
}

func (r mavenRange) String() string {
	return r.rawRange
}

func (r mavenRange) Check(v Version) bool {
	ver, ok := v.(mavenVersion)
	if !ok {
		panic(fmt.Sprintf(errWrongType, r.Type(), v.Type()))
	}

	return r.mvnRange(ver.mvnVersion)
}
