package versionarium

import (
	"encoding/json"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/semver"
)

const (
	semverType   = "semver"
	errWrongType = "cannot compare versions of different types: want %s, have %s"
)

var (
	_ Version      = (*version)(nil)
	_ VersionRange = (*versionRange)(nil)
)

type (
	version struct {
		semver *semver.Version
		verStr string
	}

	versionRange struct {
		rawConstraints string
		constraints    *semver.Constraints
	}
)

func newSemverVersion(rawVersion string, strict bool) (version, error) {
	var (
		ver *semver.Version
		err error
	)

	if strict {
		ver, err = semver.StrictNewVersion(rawVersion)
	} else {
		ver, err = semver.NewVersion(rawVersion)
	}

	if err != nil {
		return version{}, err
	}

	return version{
		semver: ver,
		verStr: ver.String(),
	}, nil
}

func newSemverRange(rawVersionRange string) (versionRange, error) {
	constraint, err := semver.NewConstraint(rawVersionRange)
	if err != nil {
		return versionRange{}, err
	}
	return versionRange{constraints: constraint, rawConstraints: rawVersionRange}, nil
}

func (v version) Type() string {
	return semverType
}

func (v version) String() string {
	return v.verStr
}

func (v version) MarshalJSON() ([]byte, error) {
	return json.Marshal(v.verStr)
}

func (v version) ReleaseInfo() string {
	return v.semver.Prerelease()
}

func (v version) BuildInfo() string {
	return v.semver.Metadata()
}

func (v version) Compare(o Version) int {
	ver, ok := o.(version)
	if !ok {
		panic(xerrors.Errorf(errWrongType, v.Type(), ver.Type()))
	}
	return v.semver.Compare(ver.semver)
}

func (v version) LessThan(o Version) bool {
	return v.Compare(o) < 0
}

func (v version) GreaterThan(o Version) bool {
	return v.Compare(o) > 0
}

func (v version) Equal(o Version) bool {
	return v.Compare(o) == 0
}

func (r versionRange) Type() string {
	return semverType
}

func (r versionRange) String() string {
	return r.rawConstraints
}

func (r versionRange) Check(v Version) bool {
	semverVersion, ok := v.(version)
	if !ok {
		panic(xerrors.Errorf(errWrongType, r.Type(), v.Type()))
	}

	return r.constraints.Check(semverVersion.semver)
}
