package versionarium

import (
	"encoding/json"
	"fmt"
	"strings"

	lru "github.com/hashicorp/golang-lru"

	"a.yandex-team.ru/library/go/core/xerrors"
)

const ErrNotSupported = "language (%s) is not supported"

type (
	Version interface {
		json.Marshaler
		Type() string
		String() string
		ReleaseInfo() string
		BuildInfo() string
		Compare(o Version) int
		LessThan(o Version) bool
		GreaterThan(o Version) bool
		Equal(o Version) bool
	}

	VersionRange interface {
		Type() string
		String() string
		Check(v Version) bool
	}
)

var (
	versionsCache = func() *lru.Cache {
		c, err := lru.New(4096)
		if err != nil {
			panic(fmt.Sprintf("can't create versions cache: %s", err.Error()))
		}
		return c
	}()
)

func NewVersion(language, rawVersion string) (Version, error) {
	rawVersion = strings.TrimSpace(rawVersion)
	cacheKey := language + "@" + rawVersion

	if c, ok := versionsCache.Get(cacheKey); ok {
		return c.(Version), nil
	}

	ver, err := newVersion(language, rawVersion)
	if err != nil {
		return ver, err
	}

	versionsCache.Add(cacheKey, ver)
	return ver, nil
}

func newVersion(language, rawVersion string) (Version, error) {
	switch language {
	case "nodejs":
		return newSemverVersion(rawVersion, true)
	case "python":
		return newSemverVersion(rawVersion, false)
	case "golang":
		return newGoSemverVersion(rawVersion)
	case "java":
		return newMavenVersion(rawVersion)
	case "cpp":
		return newCppVersion(rawVersion)
	default:
		return newMockVersion(rawVersion), xerrors.Errorf(ErrNotSupported, language)
	}
}

func MustNewVersion(language, rawVersion string) Version {
	v, err := NewVersion(language, rawVersion)
	if err != nil {
		panic(err)
	}
	return v
}

func NewRange(language, rawVersionRange string) (VersionRange, error) {
	if rawVersionRange == anyRangeStr {
		// special case
		return newAnyRange(), nil
	}

	switch language {
	case "nodejs", "python":
		return newSemverRange(rawVersionRange)
	case "golang":
		return newGoSemverRange(rawVersionRange)
	case "java":
		return newMavenRange(rawVersionRange)
	case "cpp":
		return newCppVersionRange(rawVersionRange)
	default:
		return newMockRange(rawVersionRange), xerrors.Errorf(ErrNotSupported, language)
	}
}
