package main

import (
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"net"
	"net/http"
	"net/url"
	"os"
	"os/exec"
	"strings"
	"time"
)

type (
	// Config for the plugin.
	Config struct {
		Token      string
		Commands   [][]string
		Version    string
		AutoUpdate bool
	}

	// Plugin values
	Plugin struct {
		Config Config
	}
)

const (
	yadiDownloadURL = "https://tools.sec.yandex-team.ru/api/v1/release/yadi/linux/latest"
	yadiInfoURL     = "https://tools.sec.yandex-team.ru/api/v1/release/yadi/linux/latest/info"
	yadiPath        = "/bin/yadi"
	tmpYadiPath     = "/tmp/yadi"
)

var (
	HttpClient = http.Client{
		Transport: &http.Transport{
			Proxy: http.ProxyFromEnvironment,
			DialContext: (&net.Dialer{
				Timeout:   2 * time.Second,
				KeepAlive: 30 * time.Second,
			}).DialContext,
			TLSHandshakeTimeout: 2 * time.Second,
		},
		Timeout: 20 * time.Second,
	}

	defaultArgs = []string{"--skip-version-check"}
)

// Exec executes the plugin.
func (p Plugin) Exec() error {
	if len(p.Config.Commands) == 0 {
		return nil
	}

	var err error
	if p.Config.AutoUpdate {
		err = updateYadi(p.Config.Version)
	} else {
		err = checkYadiVersion(p.Config.Version)
	}

	if err != nil {
		log.Println(err.Error())
	}

	comms := make([]*exec.Cmd, len(p.Config.Commands))
	for i, c := range p.Config.Commands {
		comms[i] = exec.Command(yadiPath, append(defaultArgs, c...)...)
	}

	return runCommands(comms, p.Config.Token)
}

func runCommands(cmds []*exec.Cmd, token string) error {
	for _, cmd := range cmds {
		err := runCommand(cmd, token)

		if err != nil {
			return err
		}
	}

	return nil
}

// trace writes each command to standard error (preceded by a ‘$ ’) before it
// is executed. Used for debugging your build.
func trace(cmd *exec.Cmd) {
	fmt.Fprintf(os.Stdout, "+ %s\n", strings.Join(cmd.Args, " "))
}

func runCommand(cmd *exec.Cmd, token string) error {
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr
	cmd.Env = os.Environ()
	cmd.Env = append(cmd.Env, fmt.Sprintf("YADI_TOKEN=%s", token))

	trace(cmd)
	return cmd.Run()
}

func updateYadi(version string) error {
	params := url.Values{
		"current": {version},
	}

	resp, err := http.Get(fmt.Sprintf("%s?%s", yadiDownloadURL, params.Encode()))
	if err != nil {
		return err
	}
	defer func() { _ = resp.Body.Close() }()
	defer func() { _, _ = io.Copy(ioutil.Discard, resp.Body) }()

	if resp.StatusCode == 304 {
		// Nothing to update
		return nil
	}

	if resp.StatusCode != 200 {
		return fmt.Errorf("Failed to download new release, server returns %d status code", resp.StatusCode)
	}

	log.Println("New Yadi version available, update started")
	out, err := os.Create(tmpYadiPath)
	if err != nil {
		return err
	}
	defer out.Close()

	_, err = io.Copy(out, resp.Body)
	if err != nil {
		return err
	}

	err = os.Chmod(tmpYadiPath, 0777)
	if err != nil {
		return err
	}

	return renameFile(tmpYadiPath, yadiPath)
}

func checkYadiVersion(current string) error {
	resp, err := http.Get(yadiInfoURL)
	if err != nil {
		return err
	}
	defer func() { _ = resp.Body.Close() }()
	defer func() { _, _ = io.Copy(ioutil.Discard, resp.Body) }()

	if resp.StatusCode != 200 {
		return fmt.Errorf("Failed to check Yadi version, server returns %d status code", resp.StatusCode)
	}

	var response struct {
		Result struct {
			System      string `json:"string"`
			Version     string `json:"version"`
			Stable      bool   `json:"stable"`
			DownloadURL string `json:"download_url"`
		}
	}

	err = json.NewDecoder(resp.Body).Decode(&response)
	if err != nil {
		return err
	}

	if response.Result.Version != current {
		log.Printf("New version available: %s\n", response.Result.Version)
	}

	return nil
}

func renameFile(src string, dst string) (err error) {
	err = copyFile(src, dst)
	if err != nil {
		return fmt.Errorf("failed to copy source file %s to %s: %s", src, dst, err)
	}
	err = os.RemoveAll(src)
	if err != nil {
		return fmt.Errorf("failed to cleanup source file %s: %s", src, err)
	}
	return nil
}

func copyFile(src, dst string) (err error) {
	in, err := os.Open(src)
	if err != nil {
		return
	}
	defer in.Close()

	out, err := os.Create(dst)
	if err != nil {
		return
	}
	defer func() {
		if e := out.Close(); e != nil {
			err = e
		}
	}()

	_, err = io.Copy(out, in)
	if err != nil {
		return
	}

	err = out.Sync()
	if err != nil {
		return
	}

	si, err := os.Stat(src)
	if err != nil {
		return
	}
	err = os.Chmod(dst, si.Mode())
	if err != nil {
		return
	}

	return
}
