package commands

import (
	"context"
	"encoding/json"
	"fmt"
	"strings"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed"
)

var (
	listCmd = &cobra.Command{
		Use:       fmt.Sprintf("ls [flags] [%s]", strings.Join(SupportedTargets, "/")),
		Short:     "List short vulnerabilities info",
		Args:      matchAll(cobra.ExactArgs(1), cobra.OnlyValidArgs),
		ValidArgs: SupportedTargets,
		RunE:      doList,
	}
	format string
)

func init() {
	flags := listCmd.PersistentFlags()
	flags.StringVar(&format, "format", "list", "Output format: list / json")
}

func doList(cmd *cobra.Command, args []string) error {
	vulnFeed, err := parseTarget(args)
	if err != nil {
		return err
	}
	simplelog.Debug("New feed created: ", "feed", vulnFeed.Name())

	ctx := context.Background()
	result, err := vulnFeed.Dump(ctx, feed.DumpingOpts{
		Range: opts.Since,
	})
	if err != nil {
		return err
	}
	simplelog.Debug("Feed dumped")

	switch format {
	case "json":
		byteJSON, err := json.MarshalIndent(result, "", "    ")
		if err != nil {
			return err
		}
		fmt.Println(string(byteJSON))

	case "list":
		for _, vulns := range result {
			for _, v := range vulns {
				fmt.Printf("%s: %s@%s [%s]\r\n", v.ID, v.Package, v.VulnerableVersions, v.Title)
			}
		}

	default:
		return xerrors.New("Not supported list format. Use list / json")
	}

	return nil
}
