package commands

import (
	"context"
	"fmt"
	"os"
	"sort"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/snatcher/internal/filters"
	"a.yandex-team.ru/security/yadi/snatcher/internal/storage"
	yadiclient "a.yandex-team.ru/security/yadi/snatcher/internal/yadi-client"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed"
	"a.yandex-team.ru/security/yadi/web/pkg/advisories"
	yadi "a.yandex-team.ru/security/yadi/yadi/pkg/feed"
)

var (
	limit                       = -1
	yadiURL, storageURL, bucket string
	force                       bool
	pushCmd                     = &cobra.Command{
		Use:   "push [flags]",
		Short: "Download feed from yadi and upload to S3",
		RunE:  doPush,
	}
)

func init() {
	flags := pushCmd.PersistentFlags()
	flags.StringVar(&yadiURL, "url", "https://yadi.yandex-team.ru/", "Yadi URL")
	flags.StringVar(&storageURL, "storage", "s3.mds.yandex.net", "S3 endpoint")
	flags.StringVar(&bucket, "bucket", "db", "S3 bucket name")
	flags.BoolVar(&force, "force", false, "Do not compare hash-sums of database before uploading")
}

func doPush(_ *cobra.Command, _ []string) error {
	now := time.Now().Unix()

	yadiClient, err := yadiclient.NewClient(yadiURL)
	if err != nil {
		return fmt.Errorf("failed to init yadi client: %w", err)
	}

	yadiStorage, err := storage.NewStorage(storage.Options{
		Endpoint:        storageURL,
		Bucket:          bucket,
		AccessKeyID:     os.Getenv("ACCESS_KEY_ID"),
		SecretAccessKey: os.Getenv("SECRET_ACCESS_KEY"),
	})
	if err != nil {
		return fmt.Errorf("failed to init storage: %w", err)
	}
	defer func() { _ = yadiStorage.Close() }()

	ctx := context.Background()
	vulns, err := yadiClient.ListModerateVulns(ctx, limit)
	if err != nil {
		return fmt.Errorf("failed to load vulns: %w", err)
	}

	sort.Sort(vulns)

	webVulns := make([]advisories.Advisory, 0)
	vulnsByLang := make(map[feed.Platform]yadi.Feed)
	for _, vuln := range vulns {
		if _, ok := feed.Platforms[vuln.Language]; !ok {
			return fmt.Errorf("unknown language %s of vuln: %s", vuln.Language, vuln.ID)
		}
		if vulnsByLang[vuln.Language] == nil {
			vulnsByLang[vuln.Language] = make(yadi.Feed, 0)
		}
		entry, err := filters.NewYadiVuln(vuln)
		if err != nil {
			return fmt.Errorf("failed to convert vuln %s: %w", vuln.ID, err)
		}
		vulnsByLang[vuln.Language] = append(vulnsByLang[vuln.Language], entry)

		w, err := filters.NewWebVuln(vuln)
		if err != nil {
			return fmt.Errorf("failed to create vuln for yadi-web: %w", err)
		}
		webVulns = append(webVulns, w)
	}

	webDatabaseName := "web.json"
	err = yadiStorage.UploadDatabase(ctx, webVulns, storage.UploadOpts{
		GzCompress: false,
		Force:      force,
		DBPath:     webDatabaseName,
		TimeStamp:  now,
	})
	if err != nil {
		return fmt.Errorf("failed to upload %s database: %w", webDatabaseName, err)
	}
	simplelog.Info("database was uploaded successfully", "db", webDatabaseName)

	for lang, langVulns := range vulnsByLang {
		rawDatabaseName := lang + ".json"
		gzDatabaseName := rawDatabaseName + ".gz"

		err = yadiStorage.UploadDatabase(ctx, langVulns, storage.UploadOpts{
			TimeStamp:  now,
			GzCompress: false,
			Force:      force,
			DBPath:     rawDatabaseName,
		})
		if err != nil {
			return fmt.Errorf("failed to upload %s database: %w", rawDatabaseName, err)
		}
		simplelog.Info("database was uploaded successfully", "db", rawDatabaseName)

		err = yadiStorage.UploadDatabase(ctx, langVulns, storage.UploadOpts{
			TimeStamp:  now,
			GzCompress: true,
			Force:      force,
			DBPath:     gzDatabaseName,
		})
		if err != nil {
			return fmt.Errorf("failed to upload %s database: %w", gzDatabaseName, err)
		}
		simplelog.Info("database was uploaded successfully", "db", gzDatabaseName)
	}

	return nil
}
