package commands

import (
	"os"
	"time"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/snatcher/internal/config"
	"a.yandex-team.ru/security/yadi/snatcher/internal/mailer"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed/alpine"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed/debian"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed/kernel"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed/snyk"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed/stub"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed/ubuntu"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed/wiki"
)

var RootCmd = &cobra.Command{
	Use:          "snatcher [action] [target]",
	SilenceUsage: true,
	Short:        "snatcher for vulnerability feed",
	Long:         `Yadi Vulnerability Feed Snatcher`,
	PersistentPreRun: func(cmd *cobra.Command, args []string) {
		if beVerbose {
			simplelog.SetLevel(simplelog.DebugLevel)
		}
	},
}

var (
	SupportedTargets = []string{
		"snyk",
		"wiki",
		"linux-ubuntu",
		"linux-debian",
		"linux-alpine",
		"linux-kernel",
	}

	beVerbose bool
	opts      struct {
		EnvType     string
		ShowVersion bool
		Action      string
		Since       int64
	}
	// https://oauth.yandex-team.ru/authorize?response_type=token&client_id=682d94c43dc34af69bc75d7315d13f6c
	YandexTeamOAuthTokenVar = os.Getenv("YADI_SNATCHER_OAUTH_TOKEN")
)

func init() {
	flags := RootCmd.PersistentFlags()
	flags.BoolVarP(&beVerbose, "verbose", "v", false, "verbose output")
	flags.Int64Var(&opts.Since, "since", 0, "After: vulnerability id / timestamp")
	flags.StringVar(&opts.EnvType, "env", "dev", "Environment type (dev/prod)")
	RootCmd.AddCommand(
		versionCmd,
		listCmd,
		updateCmd,
		pushCmd,
	)
}

func newConfig() *config.Config {
	switch opts.EnvType {
	case "prod":
		return &config.Config{
			DumpTimeout:  10 * time.Minute,
			YdbEndpoint:  "ydb-ru.yandex.net:2135",
			YdbDatabase:  "/ru/home/robot-yadi/mydb",
			YdbPath:      "/prod/feed",
			YdbAuthToken: YandexTeamOAuthTokenVar,
			Mailer: mailer.NewMailer(mailer.Opts{
				Host: "outbound-relay.yandex.net",
				Port: 25,
				To: []string{
					"yadi-feed@yandex-team.ru",
				},
			}),
		}

	case "dev":
		fallthrough

	default:
		return &config.Config{
			DumpTimeout:  10 * time.Minute,
			YdbEndpoint:  "ydb-ru.yandex.net:2135",
			YdbDatabase:  "/ru/home/robot-yadi/mydb",
			YdbPath:      "/dev/feed",
			YdbAuthToken: YandexTeamOAuthTokenVar,
			Mailer: mailer.NewMailer(mailer.Opts{
				Host: "outbound-relay.yandex.net",
				Port: 25,
				To: []string{
					"robot-yadi@yandex-team.ru",
					"buglloc@yandex-team.ru",
				},
			}),
		}
	}
}

func Execute() {
	RootCmd.Version = config.FullVersion()
	RootCmd.SetVersionTemplate("{{.Version}}\n")

	if err := RootCmd.Execute(); err != nil {
		os.Exit(1)
	}

}

func parseTarget(args []string) (feed.VulnerabilityFeed, error) {
	stubFeed, _ := stub.NewFeed()
	switch len(args) {
	case 0:
		return stubFeed, xerrors.New("empty args")
	case 1:
		// TODO(buglloc): move to feed itself?
		switch args[0] {
		case "snyk":
			return snyk.NewFeed(snyk.Opts{
				// sec-01da6pyfgf4er64bp0prvj4kns
				ConsumerID: os.Getenv("SNYK_CONSUMER_ID"),
				Secret:     os.Getenv("SNYK_SECRET"),
			})
		case "wiki":
			if YandexTeamOAuthTokenVar == "" {
				return stubFeed, xerrors.New("empty YADI_SNATCHER_OAUTH_TOKEN env var")
			}
			return wiki.NewFeed(wiki.Opts{
				OAuthToken: YandexTeamOAuthTokenVar,
			})
		case "linux-ubuntu":
			return ubuntu.NewFeed(ubuntu.Opts{
				TmpDir: os.Getenv("SNATCHER_TMP_DIR"),
			})
		case "linux-debian":
			return debian.NewFeed(debian.Opts{
				TmpDir: os.Getenv("SNATCHER_TMP_DIR"),
			})
		case "linux-alpine":
			return alpine.NewFeed(alpine.Opts{
				TmpDir: os.Getenv("SNATCHER_TMP_DIR"),
			})
		case "linux-kernel":
			return kernel.NewFeed(kernel.Opts{
				TmpDir: os.Getenv("SNATCHER_TMP_DIR"),
			})
		default:
			return stub.NewFeed()
		}
	default:
		return stub.NewFeed()
	}
}

func matchAll(checks ...cobra.PositionalArgs) cobra.PositionalArgs {
	return func(cmd *cobra.Command, args []string) error {
		for _, check := range checks {
			if err := check(cmd, args); err != nil {
				return err
			}
		}
		return nil
	}
}
