package commands

import (
	"context"
	"fmt"
	"strings"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/snatcher/internal/config"
	"a.yandex-team.ru/security/yadi/snatcher/internal/mailer"
	"a.yandex-team.ru/security/yadi/snatcher/internal/stat"
	"a.yandex-team.ru/security/yadi/snatcher/internal/ydb"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed"
)

var (
	updateCmd = &cobra.Command{
		Use:       fmt.Sprintf("update [flags] [%s]", strings.Join(SupportedTargets, "/")),
		Short:     "Update database with new vulns",
		Args:      matchAll(cobra.ExactArgs(1), cobra.OnlyValidArgs),
		ValidArgs: SupportedTargets,
		RunE:      doUpdate,
	}
	printStat bool
	mapOldIds bool
	sendEmail bool
	dry       bool
)

type UpdateDBOpts struct {
	Config    *config.Config
	Feed      *feed.VulnerabilityFeed
	Dump      feed.Result
	DeleteTTL int64
	Counter   *stat.Counter
}

func init() {
	flags := updateCmd.PersistentFlags()
	flags.BoolVar(&printStat, "count", true, "print update statistic")
	flags.BoolVar(&mapOldIds, "map", false, "replace YadiID with old IDs for excluding compatibility")
	flags.BoolVar(&sendEmail, "email", false, "send email with new vulnerabilities")
	flags.BoolVar(&dry, "dry-run", false, "do not update database")
}

func doUpdate(cmd *cobra.Command, args []string) error {
	vulnFeed, err := parseTarget(args)
	if err != nil {
		return err
	}
	simplelog.Debug("New feed created: ", "feed", vulnFeed.Name())

	cfg := newConfig()
	simplelog.Debug("Update config created")

	ctx, cancel := context.WithTimeout(context.Background(), cfg.DumpTimeout)
	defer cancel()

	result, err := vulnFeed.Dump(ctx, feed.DumpingOpts{
		Range:      opts.Since,
		ReplaceIDs: mapOldIds,
	})
	if err != nil {
		return err
	}
	simplelog.Debug("Feed dumped")

	err = UpdateDB(UpdateDBOpts{
		Config:    cfg,
		Feed:      &vulnFeed,
		Dump:      result,
		DeleteTTL: 86400, // 1 day in seconds
		Counter:   stat.NewCounter(),
	})

	return err
}

func UpdateDB(opts UpdateDBOpts) error {
	vulnerabilityFeed := *opts.Feed
	vulnsToEmail := make([]mailer.VulnWithComment, 0)

	dbOpts := ydb.Options{
		Endpoint:  opts.Config.YdbEndpoint,
		Path:      opts.Config.YdbPath,
		Database:  opts.Config.YdbDatabase,
		AuthToken: opts.Config.YdbAuthToken,
		SrcType:   vulnerabilityFeed.Name(),
	}

	database, err := ydb.New(context.Background(), &dbOpts)
	if err != nil {
		return xerrors.Errorf("failed to connect to database: %s", err)
	}
	simplelog.Debug("Database created")

	// dump src table
	dbSrcVulns, err := database.LookupVulns(vulnerabilityFeed.Name())
	if err != nil {
		return xerrors.Errorf("failed to dump vulns from database: %s", err)
	}
	simplelog.Debug("Database dumped")

	var trackedVulns []ydb.UpdateData     // need moderation
	var untrackedVulns []ydb.UpdateData   // just updating not general options, does not need moderation
	var deletedVulns []feed.Vulnerability // vulns which have gone from feed and didn't appear for opts.DeleteTTL

	for _, feedPlatformVulns := range opts.Dump {
		for _, feedVuln := range feedPlatformVulns {
			if err := feedVuln.Validate(); err != nil {
				simplelog.Warn("skip invalid feed vuln", "err", err)
				continue
			}

			platform, err := vulnerabilityFeed.GetPlatformByAlias(feedVuln.Language)
			if err != nil {
				platform = feedVuln.Language
			}

			if err := feedVuln.Adjust(); err != nil {
				return xerrors.Errorf("failed to adjust vuln: %w", err)
			}

			if dbVuln, vulnExistsInSrcTable := dbSrcVulns[feedVuln.ID]; vulnExistsInSrcTable {
				if dbVuln.StrictCompare(feedVuln) {
					// general parameters have not changed
					untrackedVulns = append(untrackedVulns, ydb.UpdateData{
						SrcType:       vulnerabilityFeed.Name(),
						Vulnerability: feedVuln,
						Platform:      platform,
					})
				} else {
					// general parameters have changed
					trackedVulns = append(trackedVulns, ydb.UpdateData{
						SrcType:       vulnerabilityFeed.Name(),
						Action:        "update",
						Vulnerability: feedVuln,
						Platform:      platform,
					})
					opts.Counter.AddChanged(1)
					vulnsToEmail = append(vulnsToEmail, mailer.VulnWithComment{
						Vuln:    feedVuln,
						Comment: "updated",
					})
				}
			} else {
				// new vuln
				trackedVulns = append(trackedVulns, ydb.UpdateData{
					SrcType:       vulnerabilityFeed.Name(),
					Action:        "new",
					Vulnerability: feedVuln,
					Platform:      platform,
				})
				opts.Counter.AddNew(1)
				vulnsToEmail = append(vulnsToEmail, mailer.VulnWithComment{
					Vuln:    feedVuln,
					Comment: "added",
				})
			}
		}
	}
	if !dry {
		err = database.UpdateActions(trackedVulns)
		if err != nil {
			return err
		}
		simplelog.Debug(`"action" table is updated`)

		err = database.UpdateSrc(append(trackedVulns, untrackedVulns...))
		if err != nil {
			return err
		}
		simplelog.Debug(`"src" table is updated`)

		deletedVulns, err = database.DeleteSrc(opts.DeleteTTL, vulnerabilityFeed.Name())
		if err != nil {
			return err
		}

		for _, vuln := range deletedVulns {
			opts.Counter.AddDeleted(1)
			vulnsToEmail = append(vulnsToEmail, mailer.VulnWithComment{
				Vuln:    vuln,
				Comment: "deleted",
			})
		}
	}

	if printStat {
		c := opts.Counter.Flush()
		simplelog.Info(fmt.Sprintf("%d vulns was added", c.New))
		simplelog.Info(fmt.Sprintf("%d vulns was changed", c.Changed))
		simplelog.Info(fmt.Sprintf("%d vulns was deleted", c.Deleted)) //TODO(melkikh): does not work with dry == true
	}

	if sendEmail && !dry {
		f := *opts.Feed
		err = opts.Config.Mailer.Send(mailer.Data{
			Feed:  f.Name(),
			Vulns: vulnsToEmail,
		})
	}

	return err
}
