package gitutil

import (
	"context"
	"io/ioutil"
	"os"
	"os/exec"
	"strconv"

	"a.yandex-team.ru/library/go/core/xerrors"
)

type Repository struct {
	remote string
	path   string
}

func NewRepo(remote, tempDirPrefix string) (Repository, error) {
	path, err := ioutil.TempDir(os.TempDir(), tempDirPrefix)
	if err != nil {
		return Repository{}, xerrors.Errorf("failed to create tempdir: %w", err)
	}

	return Repository{
		remote: remote,
		path:   path,
	}, nil
}

// Clone remote repository in temporary directory and return path to this one.
// For cloning full repository without specific depth just specify depth as 0 or any negative number.
func (r *Repository) Clone(depth int64) (string, error) {
	ctx := context.Background()
	return r.CloneWithContext(ctx, depth)
}

// Clone remote repository in temporary directory and return path to this one.
// "clone" command is executed with specified context.
// For cloning full repository without specific depth just specify depth as 0 or any negative number.
func (r *Repository) CloneWithContext(ctx context.Context, depth int64) (string, error) {
	if err := clone(ctx, depth, r.remote, r.path); err != nil {
		return "", xerrors.Errorf("failed to clone repo: %w", err)
	}
	return r.path, nil
}

func clone(ctx context.Context, depth int64, remote, path string) error {
	args := []string{
		"clone",
	}

	if depth > 0 {
		args = append(args, "--depth", strconv.FormatInt(depth, 10))
	}

	args = append(args, remote, ".")

	cmd := exec.CommandContext(ctx, "git", args...)
	cmd.Dir = path
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr

	if err := cmd.Run(); err != nil {
		_ = os.RemoveAll(path)
		return err
	}

	return nil
}

// Checkout switches repository branch to specified one.
func (r *Repository) Checkout(branch string) (string, error) {
	ctx := context.Background()
	return r.CheckoutWithContext(ctx, branch)
}

// Checkout switches repository branch to specified one.
// "checkout" command is executed with specified context.
func (r *Repository) CheckoutWithContext(ctx context.Context, branch string) (string, error) {
	cmd := exec.CommandContext(ctx, "git", "checkout", branch)
	cmd.Dir = r.path
	cmd.Stderr = os.Stderr

	if err := cmd.Run(); err != nil {
		return "", err
	}

	return r.path, nil
}

// Clean the content of repository directory.
func (r *Repository) Clean() error {
	return os.RemoveAll(r.path)
}
