package mailer

import (
	"bytes"
	"fmt"
	html "html/template"
	"net/smtp"
	"strings"
	txt "text/template"
	"time"

	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed"
)

type (
	VulnWithComment struct {
		Vuln    feed.Vulnerability
		Comment string
	}

	YadiMailer struct {
		addr   string
		from   string
		to     []string
		header *txt.Template
		body   *html.Template
	}

	Opts struct {
		Host string
		Port int
		To   []string
	}

	Data struct {
		Feed  string
		Vulns []VulnWithComment
	}

	Mail struct {
		Data
		To   string
		From string
		Date string
	}
)

var (
	mailHeader = `From: "{{ .From }}"
To: {{ .To }}
Subject: New vulnerabilities from "{{ .Feed }}" (at {{ .Date }})
Content-Type: text/html
`
	mailBodyHTML = `
<h3>{{ len .Vulns }} vulnerabilities:</h3> 
{{range $index, $data := .Vulns }}
	{{ $vuln := $data.Vuln }}<p>{{$data.Comment | capitalize }} {{ $vuln.ID }}: {{ $vuln.Title }} in <b>{{ $vuln.Package }}</b> with {{ $vuln.CvssScore }} CVSS-score.</p>
{{end}}
<br>
You can proceed to moderation on <a href="https://yadi.yandex-team.ru/moderate" target="_blank">this page</a>.<br>
<br><br>--
<br>Best Regards,
<br>

Yandex Dependency Inspector
`
	tmplFuncs = html.FuncMap{
		"capitalize": func(str string) string {
			return strings.ToUpper(str[:1]) + str[1:]
		},
	}
)

func NewMailer(opts Opts) *YadiMailer {
	m := new(YadiMailer)
	m.addr = fmt.Sprintf("%s:%d", opts.Host, opts.Port)
	m.from = "robot-yadi@yandex-team.ru"
	m.to = opts.To
	m.header = txt.Must(txt.New("mailHeader").Parse(mailHeader))
	m.body = html.Must(html.New("mailBodyHTML").Funcs(tmplFuncs).Parse(mailBodyHTML))
	return m
}

func (m *YadiMailer) Send(data Data) (err error) {
	if len(data.Vulns) == 0 {
		return nil
	}
	mail := Mail{
		Data: data,
		From: m.from,
		To:   strings.Join(m.to, ", "),
		Date: time.Now().Format("2006-01-02 15:04"),
	}

	var buf1 bytes.Buffer
	err = m.header.Execute(&buf1, mail)
	if err != nil {
		return
	}

	var buf2 bytes.Buffer
	err = m.body.Execute(&buf2, mail)
	if err != nil {
		return
	}

	msg := buf1.Bytes()
	msg = append(msg, buf2.Bytes()...)

	err = smtp.SendMail(m.addr, nil, m.from, m.to, msg)
	return
}
