package storage

import (
	"bytes"
	"compress/gzip"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"

	"a.yandex-team.ru/security/libs/go/hashreader"
	"a.yandex-team.ru/security/libs/go/yahttp"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/manifestor"
)

const (
	region       = "yandex"
	manifestPath = "manifest.json"
)

type (
	Storage struct {
		opts       Options
		s3Client   *s3.S3
		uploader   *s3manager.Uploader
		downloader *s3manager.Downloader
	}
	Options struct {
		Endpoint        string
		Bucket          string
		AccessKeyID     string
		SecretAccessKey string
	}
	UploadOpts struct {
		TimeStamp  int64
		GzCompress bool
		Force      bool
		DBPath     string
	}
)

func NewStorage(opts Options) (*Storage, error) {
	//TODO(melkikh): add logging
	storage := &Storage{
		opts: opts,
	}

	if err := storage.initS3Client(); err != nil {
		return nil, fmt.Errorf("failed to init s3 client: %w", err)
	}

	if err := storage.initBucket(); err != nil {
		return nil, fmt.Errorf("failed to initialize S3 bucket: %w", err)
	}

	storage.initUploader()

	storage.initDownloader()

	return storage, nil
}

func (s *Storage) Close() error {
	return nil
}

func (s *Storage) initS3Client() error {
	s3Credentials := credentials.NewStaticCredentials(
		s.opts.AccessKeyID,
		s.opts.SecretAccessKey,
		"",
	)
	_, err := s3Credentials.Get()
	if err != nil {
		err = fmt.Errorf("bad credentials: %w", err)
		return err
	}

	cfg := aws.NewConfig().
		WithRegion(region).
		WithEndpoint(s.opts.Endpoint).
		WithCredentials(s3Credentials).
		WithHTTPClient(yahttp.NewClient(yahttp.Config{
			RedirectPolicy: yahttp.RedirectNoFollow,
			Timeout:        time.Minute * 5,
			DialTimeout:    time.Second,
		}))

	s3Session, err := session.NewSession()
	if err != nil {
		err = fmt.Errorf("failed to create session: %w", err)
		return err
	}

	s.s3Client = s3.New(s3Session, cfg)
	return nil
}

func (s *Storage) initBucket() error {
	if _, err := s.s3Client.HeadBucket(&s3.HeadBucketInput{Bucket: aws.String(s.opts.Bucket)}); err != nil {
		_, err := s.s3Client.CreateBucket(&s3.CreateBucketInput{Bucket: aws.String(s.opts.Bucket)})
		if err != nil {
			return err
		}
	}
	return nil
}

func (s *Storage) manifestExists(ctx context.Context) (bool, error) {
	_, err := s.s3Client.HeadObjectWithContext(
		ctx,
		&s3.HeadObjectInput{
			Bucket: aws.String(s.opts.Bucket),
			Key:    aws.String(manifestPath),
		},
	)

	if err != nil {
		if aerr, ok := err.(awserr.Error); ok {
			switch aerr.Code() {
			case "NotFound":
				return false, nil
			default:
				return false, err
			}
		}
		return false, err
	}
	return true, nil
}

func (s *Storage) initUploader() {
	s.uploader = s3manager.NewUploaderWithClient(s.s3Client)
}

func (s *Storage) initDownloader() {
	s.downloader = s3manager.NewDownloaderWithClient(s.s3Client)
}

func (s *Storage) UploadManifest(ctx context.Context, manifest manifestor.Manifest) error {
	data, err := json.Marshal(manifest)
	if err != nil {
		return err
	}
	return s.upload(ctx, manifestPath, data)
}

func (s *Storage) DownloadManifest(ctx context.Context) (*manifestor.Manifest, error) {
	data, err := s.download(ctx, manifestPath)
	if err != nil {
		return nil, err
	}
	manifest := manifestor.Empty()
	if err := json.Unmarshal(data, manifest); err != nil {
		return nil, err
	}
	return manifest, nil
}

func (s *Storage) UploadDatabase(ctx context.Context, data interface{}, opts UploadOpts) error {
	force := opts.Force
	manifest := manifestor.Empty()

	manifestExists, err := s.manifestExists(ctx)
	if err != nil {
		return fmt.Errorf("failed to head manifest: %w", err)
	}
	if manifestExists {
		manifest, err = s.DownloadManifest(ctx)
		if err != nil {
			return fmt.Errorf("failed to get manifest: %w", err)
		}
	} else {
		// we should upload the new one
		force = true
	}

	r, w := io.Pipe()
	defer func() { _ = r.Close() }()

	var compressor io.Writer
	var gzipWriter *gzip.Writer
	if opts.GzCompress {
		gzipWriter = gzip.NewWriter(w)
		compressor = gzipWriter
	} else {
		compressor = w
	}

	encoder := json.NewEncoder(compressor)
	go func() {
		_ = encoder.Encode(data)
		// TODO(melkikh): ugly, the closing order matters
		if gzipWriter != nil {
			_ = gzipWriter.Close()
		}
		_ = w.Close()
	}()

	hashedR, err := hashreader.NewHashReader(r)
	if err != nil {
		return fmt.Errorf("failed ti create hash reader: %w", err)
	}

	err = s.uploadR(ctx, opts.DBPath, hashedR)
	if err != nil {
		return fmt.Errorf("failed to upload database: %w", err)
	}

	hash := hashedR.Hash()
	if force || manifest.Hashes[opts.DBPath] != hash {
		manifest.Update(opts.TimeStamp, opts.DBPath, hash)
		if err = s.UploadManifest(ctx, *manifest); err != nil {
			return fmt.Errorf("failed to upload manifest: %w", err)
		}
	}
	return nil
}

func (s *Storage) upload(ctx context.Context, path string, data []byte) error {
	return s.uploadR(ctx, path, bytes.NewReader(data))
}

func (s *Storage) uploadR(ctx context.Context, path string, r io.Reader) error {
	_, err := s.uploader.UploadWithContext(
		ctx,
		&s3manager.UploadInput{
			Bucket: aws.String(s.opts.Bucket),
			Key:    aws.String(path),
			Body:   r,
		},
	)
	return err
}

func (s *Storage) download(ctx context.Context, path string) ([]byte, error) {
	result := &aws.WriteAtBuffer{}
	_, err := s.downloader.DownloadWithContext(
		ctx,
		result,
		&s3.GetObjectInput{
			Bucket: aws.String(s.opts.Bucket),
			Key:    aws.String(path),
		},
	)

	if err != nil {
		return nil, err
	}

	return result.Bytes(), nil
}
