package yadiclient

import (
	"context"
	"crypto/tls"
	"errors"
	"fmt"
	"net/url"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed"
)

const (
	ModerateAPIPrefix = "/api/v1/moderate"
)

type YadiClient struct {
	httpc *resty.Client
}

var ErrAPINotOk = errors.New("API ok: false")

func NewClient(endpoint string) (*YadiClient, error) {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		return nil, fmt.Errorf("can't create ca pool: %w", err)
	}

	httpc := resty.New().
		SetTLSClientConfig(&tls.Config{RootCAs: certPool}).
		SetBaseURL(endpoint).
		SetRetryCount(5).
		SetRetryWaitTime(1 * time.Second).
		SetRetryMaxWaitTime(100 * time.Second).
		AddRetryCondition(func(rsp *resty.Response, err error) bool {
			return err != nil
		})

	return &YadiClient{
		httpc: httpc,
	}, nil
}

func (yc *YadiClient) ListModerateVulns(ctx context.Context, limit int) (feed.Vulnerabilities, error) {
	useLimit := func() bool {
		return limit >= 0
	}

	var vulns feed.Vulnerabilities
	var lastKey string
	for {
		var out ListModerateVulnsResponse
		var apiErr APIErr
		rsp, err := yc.httpc.R().
			SetContext(ctx).
			SetError(&apiErr).
			SetResult(&out).
			ForceContentType("application/json").
			Get(fmt.Sprintf("%s/list/all?gt=%s", ModerateAPIPrefix, lastKey))
		if err != nil {
			return nil, fmt.Errorf("request failed: %w", err)
		}

		if !rsp.IsSuccess() {
			return nil, &apiErr
		}

		if !out.Ok {
			return nil, ErrAPINotOk
		}

		vulns = append(vulns, out.Result.Vulns...)

		if out.Result.IsLast || len(out.Result.Vulns) == 0 {
			simplelog.Info("successfully received the moderated vulns", "num", len(vulns))
			return vulns, nil
		}

		gt, err := parseGtParam(out.Result.NextPage)
		if err != nil {
			return nil, fmt.Errorf("failed to parse \"next_page\" URL: %w", err)
		}

		if useLimit() && len(vulns) >= limit {
			simplelog.Info("limit was exceeded", "limit", limit, "vulns", len(vulns))
			return vulns[:limit], nil
		}

		lastKey = gt
		simplelog.Info("successfully received a part of the moderated vulns", "num", len(out.Result.Vulns), "lastKey", lastKey)
	}
}

func parseGtParam(rawURL string) (string, error) {
	nextPageURL, err := url.Parse(rawURL)
	if err != nil {
		return "", err
	}

	queries, err := url.ParseQuery(nextPageURL.RawQuery)
	if err != nil {
		return "", err
	}

	gt := queries.Get("gt")
	if gt == "" {
		return "", fmt.Errorf("empty \"gt\" parameter in URL: %s", rawURL)
	}
	return gt, nil
}
