package ydb

import (
	"context"
	"fmt"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed"
)

type (
	lookupChunkOpts struct {
		SrcType string
		LastKey uint64
		Limit   uint64
	}

	LookupResult struct {
		HasNext bool
		Lastkey uint64
		Vulns   map[string]feed.Vulnerability
	}
)

func (d *DB) LookupVulns(srcType string) (result map[string]feed.Vulnerability, resultErr error) {
	result = make(map[string]feed.Vulnerability)
	var lastKey uint64

	for {
		chunk, err := d.lookupVulnsChunk(lookupChunkOpts{
			SrcType: srcType,
			LastKey: lastKey,
			Limit:   200,
		})
		if err != nil {
			return result, err
		}

		for _, vuln := range chunk.Vulns {
			result[vuln.ID] = vuln
		}

		lastKey = chunk.Lastkey

		if !chunk.HasNext {
			return
		}
	}
}

func (d *DB) lookupVulnsChunk(opts lookupChunkOpts) (result LookupResult, resultErr error) {
	result = LookupResult{
		Vulns:   make(map[string]feed.Vulnerability),
		HasNext: true,
	}

	if opts.Limit > 1000 {
		return result, xerrors.New("Wrong lookup limit. Max: 1000")
	}

	var res *table.Result
	resultErr = table.Retry(d.ctx, d.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, d.selectAllVulnsQuery)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, roTX, table.NewQueryParameters(
				table.ValueParam("$srcType", ydb.UTF8Value(opts.SrcType)),
				table.ValueParam("$lastKey", ydb.Uint64Value(opts.LastKey)),
				table.ValueParam("$limit", ydb.Uint64Value(opts.Limit)),
			))
			return
		}),
	)

	if resultErr != nil {
		return
	}

	for res.NextSet() {
		for res.NextRow() {
			var key uint64
			// key, srcType, srcId, lang, pkgName, vulnVersions, cvssScore, title
			res.SeekItem("key")
			key = res.OUint64()

			vuln, err := feed.TableToVuln(res)
			if err != nil {
				return result, fmt.Errorf("failed to read vulnerability (key: %d) from DB: %w", key, err)
			}

			if err := vuln.Validate(); err != nil {
				return result, fmt.Errorf("failed to validate vulnerability %s (key: %d) from DB: %w", vuln.String(), key, err)
			}

			result.Lastkey = key
			result.Vulns[vuln.ID] = *vuln
		}
	}

	if res.RowCount() < int(opts.Limit) {
		result.HasNext = false
	}
	simplelog.Debug("New chunk dumped", "num", res.RowCount())

	resultErr = res.Err()
	return
}
