package ydb

import "fmt"

func selectAllVulnsQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $srcType AS Utf8;
DECLARE $lastKey AS Uint64;
DECLARE $limit AS Uint64;

SELECT
    key,
    srcType,
    srcId,
    lang,
    pkgName,
    vulnVersions,
    cvssScore,
    title,
    disclosedAt,
    richDescription,
    isDeleted,
    externalReferences,
    description,
    patchedVersions,
    patchExists,
    updatedAt,
    yaId
FROM
    src
WHERE
    srcType == $srcType AND
    key > $lastKey
ORDER BY key ASC
LIMIT $limit
;`, pathPrefix)
}

func updateActionsQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $actions AS "List<Struct<
    srcType: Utf8,
    srcId: Utf8,
    vulnAction: Utf8>>";

UPSERT INTO actions
    (srcType, srcId, vulnAction)
SELECT
    srcType, srcId, vulnAction
FROM AS_TABLE($actions)
;`, pathPrefix)
}

func updateSrcQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $updatedAt AS Int64;
DECLARE $vulns AS "List<Struct<
    srcType: Utf8,
    srcId: Utf8,

	cvssScore: Float,
	vulnVersions: Utf8,
	title: Utf8,
	pkgName: Utf8,
	externalReferences: Json,
	description: Utf8,
	lang: Utf8,
	patchExists: Bool,
	patchedVersions: Utf8,
	disclosedAt: Int64,
	richDescription: Bool,
	yaId: Utf8>>";

REPLACE INTO src
    (
		key,
		srcType,
		srcId,
		cvssScore,
		vulnVersions,
		title,
		pkgName,
		externalReferences,
		description,
		lang,
		patchExists,
		patchedVersions,
		updatedAt,
		disclosedAt,
		richDescription,
		isDeleted,
		yaId
	)
SELECT
    Digest::CityHash(YQL::Concat(srcType, srcId)),
	srcType,
	srcId,
	cvssScore,
	vulnVersions,
	title,
	pkgName,
	externalReferences,
	description,
	lang,
	patchExists,
	patchedVersions,
	$updatedAt,
	disclosedAt,
	richDescription,
	False,
	yaId
FROM AS_TABLE($vulns)
;`, pathPrefix)
}

func updateDeleteActionsQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $ttl as Int64;
DECLARE $timestamp as Int64;
DECLARE $srcType as Utf8;

$action = Utf8("delete");

-- find vulns (in src) that have not been updated for a $ttl time
$toDelete = (
SELECT
    key, srcType, srcId
FROM
    src
WHERE
    updatedAt < $timestamp - $ttl AND
    srcType = $srcType AND
    isDeleted == False
);

-- mark them as "delete" in actions (for yadi-web)
UPSERT INTO actions
    (srcType, srcId, vulnAction)
SELECT
    srcType, srcId, $action as vulnAction
FROM $toDelete;

$result = (SELECT
    s.key AS key,
    s.srcType AS srcType,
    s.srcId AS srcId,
    s.lang AS lang,
    s.pkgName AS pkgName,
    s.vulnVersions AS vulnVersions,
    s.cvssScore AS cvssScore,
    s.title AS title
FROM src AS s
INNER JOIN $toDelete AS d ON (d.srcType == s.srcType) AND (d.srcId == s.srcId)
);

-- mark them as "delete" in src (we've seen that)
UPSERT INTO src
    (key, srcType, srcId, isDeleted)
SELECT
    key, srcType, srcId, True
FROM
    $toDelete
;

-- return deleted vulns
SELECT
    key, srcType, srcId, lang, pkgName, vulnVersions, cvssScore, title
FROM
    $result
;`, pathPrefix)
}
