package ydb

import (
	"context"
	"encoding/json"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed"
)

func (d *DB) UpdateActions(data []UpdateData) (resultErr error) {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	actions := make([]ydb.Value, len(data))

	for i, action := range data {
		actions[i] = ydb.StructValue(
			ydb.StructFieldValue("srcType", ydb.UTF8Value(action.SrcType)),
			ydb.StructFieldValue("srcId", ydb.UTF8Value(action.Vulnerability.ID)),
			ydb.StructFieldValue("vulnAction", ydb.UTF8Value(action.Action)),
		)
	}

	if len(actions) > 0 {

		commits, err := chunkCommit(actions, d.commitSize)
		if err != nil {
			return err
		}

		for _, commit := range commits {
			ydbActions := ydb.ListValue(commit...)
			err := table.Retry(d.ctx, d.sp,
				table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
					stmt, err := s.Prepare(ctx, d.updateActionsQuery)
					if err != nil {
						return err
					}

					_, _, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
						table.ValueParam("$actions", ydbActions),
					))
					return err
				}),
			)
			if err != nil {
				simplelog.Error("Failed to commit actions")
				return err
			}
		}
	}

	return nil
}

func (d *DB) UpdateSrc(data []UpdateData) (resultErr error) {
	if len(data) == 0 {
		return nil
	}
	now := time.Now().Unix()
	vulns := make([]ydb.Value, len(data))
	for i, vuln := range data {
		if _, err := json.Marshal(vuln.Vulnerability.References); err != nil {
			simplelog.Error("Failed to marshall references", "err", err)
			vuln.Vulnerability.References = nil
		}
		vulns[i] = feed.VulnToSrcTable(vuln.Vulnerability)
	}

	commits, err := chunkCommit(vulns, d.commitSize)
	if err != nil {
		return err
	}

	for _, commit := range commits {
		ydbVulns := ydb.ListValue(commit...)
		err := table.Retry(d.ctx, d.sp,
			table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
				stmt, err := s.Prepare(ctx, d.updateSrcQuery)
				if err != nil {
					return err
				}

				_, _, err = stmt.Execute(ctx, rwTX, table.NewQueryParameters(
					table.ValueParam("$vulns", ydbVulns),
					table.ValueParam("$updatedAt", ydb.Int64Value(now)),
				))
				return err
			}),
		)
		if err != nil {
			simplelog.Error("Failed to commit src", "err", err)
			return err
		}
	}
	return nil
}
