package alpine

import (
	"context"
	"fmt"
	"sort"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/nvd"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/debversion"
)

const (
	trackerURI = "https://secdb.alpinelinux.org/"
	nvdURI     = "https://nvd.nist.gov/vuln/detail/%s"

	packageKindMain      = "main"
	packageKindCommunity = "community"

	feedName = "alpine-aports"
)

var (
	platforms = map[feed.Platform]string{
		feed.AlpinePlatform: feed.AlpinePlatform,
	}

	packageKinds = []string{
		packageKindMain,
		packageKindCommunity,
	}

	actualBranches = []string{
		"v3.15",
		"v3.14",
		"v3.13",
	}

	branchesOrder = map[string]int{
		"v3.13": 1,
		"v3.14": 2,
		"v3.15": 3,
	}

	malformedVerReplacer = strings.NewReplacer(
		"_p", "-p",
		".-", "-",
		".r", "-r",
		"_alpha", "-alpha",
		"_beta", "-beta",
		"_rc", "-rc",
	)
)

type (
	Opts struct {
		TmpDir string
	}

	Feed struct {
		httpc  *resty.Client
		tmpDir string
	}

	SecDBResp struct {
		Packages []SecDBPackage `json:"packages"`
	}

	SecDBPackage struct {
		Pkg Package `json:"pkg"`
	}

	Package struct {
		Name     string              `json:"name"`
		SecFixes map[string][]string `json:"secfixes"`
	}
)

func NewFeed(opts Opts) (*Feed, error) {
	return &Feed{
		httpc:  resty.New().SetRetryCount(5).SetBaseURL(trackerURI),
		tmpDir: opts.TmpDir,
	}, nil
}

func (f *Feed) Name() string {
	return feedName
}

func (f *Feed) GetPlatformByAlias(alias string) (feed.Platform, error) {
	for p, a := range platforms {
		if a == alias {
			return p, nil
		}
	}
	return "", xerrors.New("not supported platform")
}

func (f *Feed) Dump(ctx context.Context, opts feed.DumpingOpts) (feed.Result, error) {
	nvdFeed, err := nvd.ParseFeed(ctx)
	if err != nil {
		return nil, xerrors.Errorf("failed to download NVD Feed: %w", err)
	}

	vulns := map[string]*vulnerability{}
	for _, branch := range actualBranches {
		if err := f.parseBranch(branch, vulns); err != nil {
			return nil, err
		}
	}

	feedVulns := map[feed.VulnID]feed.Vulnerability{}
	for _, vuln := range vulns {
		nvdInfo, ok := nvdFeed[vuln.CVE]
		if !ok {
			simplelog.Warn("skip non-existent CVE", "cve_id", vuln.CVE)
			continue
		}

		if nvdInfo.Score < 1.0 {
			simplelog.Warn("skip not interesting CVE",
				"cve_id", vuln.CVE,
				"impact_score", nvdInfo.Score)
			continue
		}

		feedVulns[vuln.ID] = feed.Vulnerability{
			ID:          strings.ToUpper(vuln.ID),
			YadiID:      strings.ToUpper(fmt.Sprintf("yadi-%s-%s", feed.AlpinePlatform, vuln.ID)),
			Package:     vuln.Package,
			SrcType:     feedName,
			Title:       fmt.Sprintf("%s (%s)", vuln.Package, vuln.CVE),
			Description: nvdInfo.Description,
			DisclosedAt: nvdInfo.PublishedDate.Unix(),
			References: []feed.Reference{
				{
					URL:   fmt.Sprintf(nvdURI, vuln.CVE),
					Title: fmt.Sprintf("NVD - %s", vuln.CVE),
				},
			},
			CvssScore:          nvdInfo.Score,
			Language:           feed.AlpinePlatform,
			VulnerableVersions: vuln.Versions.String(),
		}
	}

	return feed.Result{feed.AlpinePlatform: feedVulns}, nil
}

func (f *Feed) parseBranch(branch string, vulns map[string]*vulnerability) error {
	for _, kind := range packageKinds {
		rsp, err := f.fetchFeed(branch, kind)
		if err != nil {
			return fmt.Errorf("failed to fetch feed from branch %s with kind %s: %w", branch, kind, err)
		}

		for _, pkg := range rsp.Packages {
			err = parseFixes(branch, pkg.Pkg, vulns)
			if err != nil {
				return xerrors.Errorf("failed to parse sec fixes for pkg %s from branch %s with kind %s: %w", pkg.Pkg.Name, branch, kind, err)
			}
		}
	}

	return nil
}

func (f *Feed) fetchFeed(branch, kind string) (*SecDBResp, error) {
	var out SecDBResp
	rsp, err := f.httpc.R().
		SetResult(&out).
		SetPathParam("branch", branch).
		SetPathParam("kind", kind).
		Get("/{branch}/{kind}.json")

	if err != nil {
		return nil, err
	}

	if !rsp.IsSuccess() {
		return nil, fmt.Errorf("non-200 status: %s", rsp.Status())
	}

	return &out, nil
}

func parseFixes(branch string, pkg Package, vulns map[string]*vulnerability) error {
	// First of all - we must sort pkg versions
	var versions debversion.Versions
	secFixes := pkg.SecFixes
	for ver, cves := range secFixes {
		pkgVersion := malformedVerReplacer.Replace(ver)
		debVersion, err := debversion.NewVersion(pkgVersion)
		if err != nil {
			simplelog.Error("failed to parse version", "version", ver, "err", err.Error())
			continue
		}

		if ver != pkgVersion {
			delete(secFixes, ver)
			secFixes[pkgVersion] = cves
		}

		versions = append(versions, debVersion)
	}

	sort.Sort(versions)

	// Then - iterate over it, don't use map keys
	for _, ver := range versions {
		pkgVersion := ver.Original()
		cves, ok := secFixes[pkgVersion]
		if !ok {
			simplelog.Error("failed to find version", "version", pkgVersion)
			continue
		}

		for _, cve := range cves {
			cveID := fixCVEID(cve)
			if cveID == "" {
				continue
			}

			id := fmt.Sprintf("%s-%s", pkg.Name, strings.TrimPrefix(cveID, "CVE-"))
			if _, ok := vulns[id]; !ok {
				vulns[id] = &vulnerability{
					ID:      id,
					Package: pkg.Name,
					CVE:     cveID,
					Versions: versionsMap{
						branch: pkgVersion,
					},
				}
			} else {
				vulns[id].Versions[branch] = pkgVersion
			}
		}
	}

	return nil
}

func fixCVEID(cveID string) string {
	parts := strings.Split(cveID, " ")
	for _, part := range parts {
		if strings.HasPrefix(part, "CVE-") {
			return part
		}
	}

	return ""
}
