package debian

import (
	"context"
	"fmt"
	"strings"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/nvd"
	"a.yandex-team.ru/security/yadi/libs/osreleases"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed"
)

const (
	trackerURI   = "https://security-tracker.debian.org/tracker/data/json"
	cveURL       = "https://security-tracker.debian.org/tracker/%s"
	debianBugURL = "https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=%d"
	feedName     = "debian-security-tracker"
)

var (
	platforms = map[feed.Platform]string{
		feed.DebianPlatform: feed.DebianPlatform,
	}

	acceptableReleases = map[osreleases.Debian]struct{}{
		osreleases.DebianSid:      {},
		osreleases.DebianBuster:   {},
		osreleases.DebianBullseye: {},
		osreleases.DebianBookworm: {},
	}

	oldReleases = []osreleases.Debian{
		osreleases.DebianStretch,
		osreleases.DebianJessie,
		osreleases.DebianWheezy,
		osreleases.DebianSqueeze,
	}

	stableReleases = []osreleases.Debian{
		osreleases.DebianBuster,
		osreleases.DebianBullseye,
	}
)

type (
	Opts struct {
		TmpDir string
	}

	Feed struct {
		tmpDir string
	}
)

func NewFeed(opts Opts) (Feed, error) {
	return Feed{
		tmpDir: opts.TmpDir,
	}, nil
}

func (f Feed) Name() string {
	return feedName
}

func (f Feed) GetPlatformByAlias(alias string) (feed.Platform, error) {
	for p, a := range platforms {
		if a == alias {
			return p, nil
		}
	}
	return "", xerrors.New("not supported platform")
}

func (f Feed) Dump(ctx context.Context, opts feed.DumpingOpts) (feed.Result, error) {
	debianFeed, err := downloadFeed(ctx)
	if err != nil {
		return nil, xerrors.Errorf("failed to download debian feed: %w", err)
	}

	nvdFeed, err := nvd.ParseFeed(ctx)
	if err != nil {
		return nil, xerrors.Errorf("failed to download NVD Feed: %w", err)
	}

	eol := buildEol()
	vulns := map[feed.VulnID]feed.Vulnerability{
		eol.ID: eol,
	}

	for pkgName, cves := range debianFeed {
		if pkgName == "linux" || strings.HasPrefix(pkgName, "linux-") {
			// skip Linux kernels
			continue
		}

		if err := process(pkgName, cves, nvdFeed, vulns); err != nil {
			return nil, err
		}
	}
	return feed.Result{feed.DebianPlatform: vulns}, nil
}

func downloadFeed(ctx context.Context) (debianFeed, error) {
	simplelog.Info("request debian feed", "url", trackerURI)

	var parsedFeed debianFeed
	rsp, err := resty.New().R().
		SetContext(ctx).
		SetResult(&parsedFeed).
		Get(trackerURI)

	if err != nil {
		return nil, err
	}

	if !rsp.IsSuccess() {
		return nil, xerrors.Errorf("non-success status code: %d", rsp.StatusCode())
	}
	return parsedFeed, nil
}

func buildEol() feed.Vulnerability {
	description := `## Overview
Using a obsolete Debian release doesn't guarantee its security and stability.
You should upgrade to the nearest stable release:
`
	for _, release := range stableReleases {
		description += fmt.Sprintf("  * %s\n", strings.Title(release.String()))
	}

	vulnerableVersions := make(debianVersions, len(oldReleases))
	for i, release := range oldReleases {
		vulnerableVersions[i] = debianVersion{
			Release: release,
			Vesion:  "*",
		}
	}

	return feed.Vulnerability{
		Title:              "Obsolete Debian Release",
		Package:            "debian",
		ID:                 "OBSOLETE",
		DisclosedAt:        time.Now().Unix(),
		YadiID:             strings.ToUpper(fmt.Sprintf("yadi-%s-obsolete", feed.DebianPlatform)),
		SrcType:            feedName,
		Description:        description,
		RichDescription:    true,
		Language:           feed.DebianPlatform,
		VulnerableVersions: vulnerableVersions.String(),
		CvssScore:          9.9,
		References: []feed.Reference{
			{
				Title: "Debian Long Term Support",
				URL:   "https://wiki.debian.org/LTS",
			},
			{
				Title: "Debian Releases",
				URL:   "https://wiki.debian.org/DebianReleases",
			},
		},
	}
}

func process(pkgName string, cves debianCVEs, nvdFeed nvd.Feed, vulnerabilities map[feed.VulnID]feed.Vulnerability) error {
	for cveID, cveInfo := range cves {
		nvdCve, ok := nvdFeed[cveID]
		if !ok {
			simplelog.Info("skip non-existent (in NVD) CVE", "cve_id", cveID)
			continue
		}

		var versions debianVersions
		var vulnCVSSScore float32 = 0.0
		for rawRelease, releaseInfo := range cveInfo.Releases {
			osRelease := osreleases.DebianFromString(rawRelease)
			if osRelease == osreleases.DebianUnknown {
				simplelog.Warn("skip non-parsable debian release",
					"cve_id", cveID,
					"release", rawRelease,
				)
				continue
			}

			if _, ok := acceptableReleases[osRelease]; !ok {
				// not interesting
				simplelog.Info("skip non-actual debian release",
					"cve_id", cveID,
					"release", osRelease,
				)
				continue
			}

			var cvssScore float32
			if releaseInfo.Urgency == unassignedUrgency {
				// in case if debian security team not assigned urgency - use NVD info
				cvssScore = nvdCve.Score
			} else {
				cvssScore = releaseInfo.Urgency.CVSSScore()
			}

			if cvssScore < 1.0 {
				// no needed
				continue
			}

			if cvssScore > vulnCVSSScore {
				vulnCVSSScore = cvssScore
			}

			var fixedVersion string
			if releaseInfo.Status == fixStatusResolved && releaseInfo.FixedVersion != "" {
				fixedVersion = "<" + releaseInfo.FixedVersion
			} else {
				fixedVersion = "*"
			}

			versions = append(versions, debianVersion{
				Release: osRelease,
				Vesion:  fixedVersion,
			})
		}

		if len(versions) == 0 || vulnCVSSScore < 1.0 {
			// no actual releases affected
			simplelog.Info("skip CVE (by debian)", "cve_id", cveID)
			continue
		}

		vulnID := fmt.Sprintf("%s-%s", pkgName, strings.TrimPrefix(cveID, "CVE-"))
		yadiID := fmt.Sprintf("yadi-%s-%s", feed.DebianPlatform, vulnID)
		references := []feed.Reference{
			{
				Title: "Debian Security Bug Tracker",
				URL:   fmt.Sprintf(cveURL, cveID),
			},
		}

		if cveInfo.DebianBug > 0 {
			references = append([]feed.Reference{
				{
					Title: "Debian bug tracking system",
					URL:   fmt.Sprintf(debianBugURL, cveInfo.DebianBug),
				},
			}, references...)
		}

		description := strings.TrimSpace(cveInfo.Description)
		if description == "" {
			description = strings.TrimSpace(nvdCve.Description)
		}

		if description == "" {
			description = "(no description)"
		}

		vulnerabilities[vulnID] = feed.Vulnerability{
			ID:                 strings.ToUpper(vulnID),
			YadiID:             strings.ToUpper(yadiID),
			SrcType:            feedName,
			Title:              fmt.Sprintf("%s (%s)", pkgName, cveID),
			Description:        description,
			DisclosedAt:        nvdCve.PublishedDate.Unix(),
			References:         references,
			CvssScore:          vulnCVSSScore,
			Package:            pkgName,
			Language:           feed.DebianPlatform,
			VulnerableVersions: versions.String(),
		}
	}
	return nil
}
