package debian

import (
	"encoding/json"
	"fmt"
	"sort"
	"strings"

	"a.yandex-team.ru/security/yadi/libs/osreleases"
)

type debianRelease struct {
	Status       fixStatus `json:"status"`
	FixedVersion string    `json:"fixed_version"`
	Urgency      urgency   `json:"urgency"`
}

type debianFeedEntry struct {
	DebianBug   uint64                   `json:"debianbug"`
	Description string                   `json:"description"`
	Releases    map[string]debianRelease `json:"releases"`
}

type debianCVEs map[string]debianFeedEntry

type debianFeed map[string]debianCVEs

type fixStatus uint8

const (
	fixStatusUndetermined fixStatus = 0
	fixStatusOpen         fixStatus = 1
	fixStatusResolved     fixStatus = 2
)

func (f *fixStatus) UnmarshalJSON(data []byte) error {
	var status string
	if err := json.Unmarshal(data, &status); err != nil {
		return err
	}

	switch status {
	case "open":
		*f = fixStatusOpen
	case "resolved":
		*f = fixStatusResolved
	case "undetermined":
		*f = fixStatusUndetermined
	default:
		return fmt.Errorf("unknown fix status: %q", status)
	}

	return nil
}

type urgency float32

const (
	unassignedUrgency = 2.8
)

func (u *urgency) UnmarshalJSON(data []byte) error {
	var urgency string
	if err := json.Unmarshal(data, &urgency); err != nil {
		return err
	}

	switch urgency {
	case "end-of-life", "unimportant":
		*u = 0.0
	case "not yet assigned":
		// TODO(buglloc): WTF?!
		*u = unassignedUrgency
	case "low":
		*u = 3.0
	case "medium":
		*u = 5.0
	case "high":
		*u = 8.0
	default:
		return fmt.Errorf("unknown urgency: %q", urgency)
	}

	return nil
}

func (u urgency) CVSSScore() float32 {
	return float32(u)
}

type debianVersion struct {
	Release osreleases.Debian
	Vesion  string
}

type debianVersions []debianVersion

func (v debianVersions) Len() int {
	return len(v)
}

func (v debianVersions) Less(i, j int) bool {
	return v[j].Release < v[i].Release
}

func (v debianVersions) Swap(i, j int) {
	v[i], v[j] = v[j], v[i]
}

func (v debianVersions) String() string {
	sort.Sort(v)
	var out strings.Builder
	for _, info := range v {
		out.WriteString(info.Release.String())
		out.WriteByte(':')
		out.WriteString(info.Vesion)
		out.WriteByte(' ')
	}

	return strings.TrimSpace(out.String())
}
