package feed

import (
	"encoding/json"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/library/go/core/xerrors"
)

func TableToVuln(t *table.Result) (*Vulnerability, error) {
	var vuln = new(Vulnerability)
	var err error

	t.SeekItem("srcType")
	vuln.SrcType = t.OUTF8()

	t.SeekItem("srcId")
	vuln.ID = t.OUTF8()

	t.SeekItem("pkgName")
	vuln.Package = t.OUTF8()

	t.SeekItem("lang")
	vuln.Language = t.OUTF8()

	t.SeekItem("cvssScore")
	vuln.CvssScore = t.OFloat()

	t.SeekItem("vulnVersions")
	vuln.VulnerableVersions = t.OUTF8()

	t.SeekItem("title")
	vuln.Title = t.OUTF8()

	t.SeekItem("patchedVersions")
	vuln.PatchedVersions = t.OUTF8()

	t.SeekItem("description")
	vuln.Description = t.OUTF8()

	t.SeekItem("patchExists")
	vuln.PatchExists = t.OBool()

	t.SeekItem("externalReferences")
	extRefData := []byte(t.OJSON())
	if len(extRefData) > 0 {
		err = json.Unmarshal([]byte(t.OJSON()), &vuln.References)
		if err != nil {
			return nil, xerrors.Errorf("failed to unmarshal JSON for %s vuln: %w", vuln.ID, err)
		}
	}

	t.SeekItem("disclosedAt")
	vuln.DisclosedAt = t.OInt64()

	t.SeekItem("richDescription")
	vuln.RichDescription = t.OBool()

	t.SeekItem("yaId")
	vuln.YadiID = t.OUTF8()
	return vuln, nil
}

func VulnToFeedTable(vuln Vulnerability) ydb.Value {
	return ydb.StructValue(
		ydb.StructFieldValue("srcType", ydb.UTF8Value(vuln.SrcType)),
		ydb.StructFieldValue("srcId", ydb.UTF8Value(vuln.ID)),
		ydb.StructFieldValue("cvssScore", ydb.FloatValue(vuln.CvssScore)),
		ydb.StructFieldValue("vulnVersions", ydb.UTF8Value(vuln.VulnerableVersions)),
		ydb.StructFieldValue("title", ydb.UTF8Value(vuln.Title)),
		ydb.StructFieldValue("pkgName", ydb.UTF8Value(vuln.Package)),
		ydb.StructFieldValue("lang", ydb.UTF8Value(vuln.Language)),
	)
}

func VulnToSrcTable(vuln Vulnerability) ydb.Value {
	extRefs, _ := json.Marshal(vuln.References)
	// TODO(melkikh): handle error (error handles in caller:
	// https://a.yandex-team.ru/arc/trunk/arcadia/security/yadi/snatcher/internal/ydb/update.go?rev=r6784942#L69-70
	return ydb.StructValue(
		ydb.StructFieldValue("srcType", ydb.UTF8Value(vuln.SrcType)),
		ydb.StructFieldValue("srcId", ydb.UTF8Value(vuln.ID)),
		ydb.StructFieldValue("cvssScore", ydb.FloatValue(vuln.CvssScore)),
		ydb.StructFieldValue("vulnVersions", ydb.UTF8Value(vuln.VulnerableVersions)),
		ydb.StructFieldValue("title", ydb.UTF8Value(vuln.Title)),
		ydb.StructFieldValue("pkgName", ydb.UTF8Value(vuln.Package)),
		ydb.StructFieldValue("lang", ydb.UTF8Value(vuln.Language)),
		ydb.StructFieldValue("disclosedAt", ydb.Int64Value(vuln.DisclosedAt)),
		ydb.StructFieldValue("richDescription", ydb.BoolValue(vuln.RichDescription)),
		ydb.StructFieldValue("externalReferences", ydb.JSONValue(string(extRefs))),
		ydb.StructFieldValue("description", ydb.UTF8Value(vuln.Description)),
		ydb.StructFieldValue("patchedVersions", ydb.UTF8Value(vuln.PatchedVersions)),
		ydb.StructFieldValue("patchExists", ydb.BoolValue(vuln.PatchExists)),
		ydb.StructFieldValue("yaId", ydb.UTF8Value(vuln.YadiID)),
	)
}
