package snyk

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"github.com/golang-jwt/jwt/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/retry"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/libs/go/yahttp"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed/snyk/vulnparser"
)

const snykURI = "https://snyk.io/partners/api/v4/vulndb/feed.json"

type (
	Opts struct {
		ConsumerID string
		Secret     string
	}

	Feed struct {
		consumerID string
		secret     []byte
		fetcher    *http.Client
	}

	AuthClaims struct {
		IssuedAt int64  `json:"iat"`
		Issuer   string `json:"iss"`
		jwt.StandardClaims
	}
)

var (
	platforms = map[feed.Platform]string{
		feed.NodeJSPlatform: "js",
		feed.PythonPlatform: "python",
		feed.GolangPlatform: "golang",
		feed.JavaPlatform:   "java",
	}
)

func NewFeed(opts Opts) (Feed, error) {
	if opts.ConsumerID == "" || opts.Secret == "" {
		return Feed{}, xerrors.New("empty consumer-ID / secret parameters")
	}

	return Feed{
		consumerID: opts.ConsumerID,
		secret:     []byte(opts.Secret),
		fetcher: yahttp.NewClient(yahttp.Config{
			RedirectPolicy: yahttp.RedirectFollow,
		}),
	}, nil
}

func (f Feed) Name() string {
	return "snyk"
}

func (f Feed) Dump(ctx context.Context, opts feed.DumpingOpts) (feed.Result, error) {
	request, err := http.NewRequestWithContext(ctx, "GET", snykURI, nil)
	if err != nil {
		simplelog.Error("Failed to create request", "err", err)
		return feed.Result{}, err
	}

	since := time.Unix(opts.Range, 0).Format(time.RFC1123)
	request.Header.Add("If-Modified-Since", since)

	claims := &AuthClaims{
		IssuedAt: time.Now().Unix(),
		Issuer:   f.consumerID,
	}
	token := jwt.NewWithClaims(jwt.SigningMethodHS256, claims)
	tokenString, err := token.SignedString(f.secret)
	if err != nil {
		simplelog.Error("Failed to sign token", "err", err)
		return feed.Result{}, err
	}
	request.Header.Add("Authorization", tokenString)

	retrier := retry.New(
		retry.WithAttempts(5),
	)

	var r *http.Response
	var body []byte
	err = retrier.Try(context.Background(), func(ctx context.Context) error {
		r, err = f.fetcher.Do(request)
		if err != nil {
			simplelog.Error("Failed to fetch feed", "err", err)
			return err
		}

		defer yahttp.GracefulClose(r.Body)

		if r.StatusCode != http.StatusOK {
			var bodyString string
			bodyBytes, err := ioutil.ReadAll(r.Body)
			if err == nil {
				bodyString = string(bodyBytes)
			}
			simplelog.Warn("Bad API response", "response", fmt.Sprintf("%d: %s", r.StatusCode, bodyString))
			return xerrors.New("feed not available now")
		}

		body, err = ioutil.ReadAll(r.Body)
		if err != nil {
			return xerrors.Errorf("failed to read feed body: %w", err)
		}
		return nil
	})

	if err != nil {
		simplelog.Error("Failed to fetch feed", "err", err)
		return feed.Result{}, err
	}

	data := make(map[string][]vulnparser.Vulnerability)
	err = json.Unmarshal(body, &data)
	if err != nil {
		simplelog.Error("Failed to decode feed JSON", "err", err)
		return feed.Result{}, err
	}

	result := feed.Result{}
	for lang, vulns := range data {
		platform, err := f.GetPlatformByAlias(lang)
		if err != nil {
			continue
		}

		result[platform] = make(map[feed.VulnID]feed.Vulnerability)
		for _, vuln := range vulns {
			v, err := vulnparser.NewVuln(vuln, opts.ReplaceIDs)
			if err != nil {
				simplelog.Error(err.Error(), "platform", platform, "id", vuln.ID)
				continue
			}

			err = v.Validate()
			if err != nil {
				simplelog.Error(err.Error(), "platform", platform, "id", vuln.ID)
				continue
			}
			result[platform][vuln.ID] = *v
		}
	}

	if len(platforms) != len(result) {
		return feed.Result{}, xerrors.Errorf("Bad platforms number. Expected %d. Received: %d", len(platforms), len(result))
	}

	return result, nil
}

func (f Feed) GetPlatformByAlias(alias string) (feed.Platform, error) {
	for p, a := range platforms {
		if a == alias {
			return p, nil
		}
	}
	return "", xerrors.New("not supported platform")
}
