package vulnparser

import (
	"fmt"
	"regexp"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/snatcher/internal/mapping"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed"
)

type (
	Vulnerability struct {
		CreationTime       time.Time        `json:"creationTime"`
		DisclosureTime     time.Time        `json:"disclosureTime"`
		ModificationTime   time.Time        `json:"modificationTime"`
		PublicationTime    time.Time        `json:"publicationTime"`
		Credit             []string         `json:"credit"`
		CVEs               []string         `json:"cves"`
		CWEs               []string         `json:"cwes"`
		References         []feed.Reference `json:"references"`
		HashesRange        []string         `json:"hashesRange"`
		VulnerableVersions []string         `json:"vulnerableVersions"`
		PatchedVersions    []string         `json:"initiallyFixedIn"`
		ID                 string           `json:"id"`
		Title              string           `json:"title"`
		Package            string           `json:"package"`
		CVSSv3             string           `json:"cvssV3"`
		Description        string           `json:"description"`
		Exploit            string           `json:"exploit"`
		Language           string           `json:"language"`
		Registry           string           `json:"registry"`
		Severity           string           `json:"severity"`
		URL                string           `json:"url"`
		CvssScore          float32          `json:"cvssScore"`
		Malicious          bool             `json:"malicious"`
		Fixable            bool             `json:"fixable"`
		PatchExists        bool             `json:"patchExists"`
	}
)

var YadiIDPkgBadChars = regexp.MustCompile("[^a-zA-Z0-9]+")

func NewVuln(snyk Vulnerability, replaceID bool) (*feed.Vulnerability, error) {
	v := &feed.Vulnerability{
		ID:              snyk.ID,
		Title:           snyk.Title,
		Package:         snyk.Package,
		Language:        languages[snyk.Language],
		SrcType:         "snyk",
		RichDescription: true, // Snyk provides markdown-encoded vulnerability description
		DisclosedAt:     snyk.CreationTime.Unix(),
		CvssScore:       snyk.CvssScore,
		References: append(snyk.References, feed.Reference{
			Title: "Snyk",
			URL:   snyk.URL,
		}),
		// `fixable` is a boolean field which states whether the vulnerability can be fixed by upgrade or by patch.
		// `patchExists` is a boolean field which states whether there is a patch available in snyk product
		// (by running `snyk protect` - https://snyk.io/docs/fixing-vulnerabilities/#patches).
		PatchExists:     snyk.Fixable,
		PatchedVersions: strings.Join(snyk.PatchedVersions, ", "),
		Description:     snyk.Description,
	}

	yadiID := fmt.Sprintf("YADI-%s-%s", strings.ToUpper(v.Language), yadiIDSuffix(snyk))
	if replaceID {
		v.YadiID = mapping.Replace(v.ID, yadiID)
	} else {
		v.YadiID = yadiID
	}

	parser, ok := parsers[snyk.Language]
	if !ok {
		return nil, xerrors.Errorf("no parser for Snyk language: %s", snyk.Language)
	}

	err := parser.Parse(v, snyk)
	if err != nil {
		return nil, xerrors.Errorf("failed to parse Snyk vulnerability '%s': %w", snyk.ID, err)
	}

	return v, nil
}

func yadiIDSuffix(snyk Vulnerability) string {
	normalizedPkgName := YadiIDPkgBadChars.ReplaceAllLiteralString(snyk.Package, "-")
	normalizedPkgName = strings.Trim(normalizedPkgName, "-")
	if len(normalizedPkgName) < 2 {
		simplelog.Error(fmt.Sprintf("Can't normalize package name: too short [%s]", snyk.ID))
		return snyk.ID
	}

	split := strings.Split(snyk.ID, "-")
	id, err := strconv.ParseUint(split[len(split)-1], 10, 64)
	if err != nil {
		simplelog.Error(fmt.Sprintf("Can't normalize package name: bad ID number [%s]", snyk.ID))
		return snyk.ID
	}

	return strings.ToUpper(normalizedPkgName + "-" + strconv.FormatUint(id, 10))
}

func processEdgeEffects(semverConstraint string) string {
	switch semverConstraint {
	case "<=0.0.0", "<0.0.0", "":
		return "<0.0.0"
	case ">0.0.0", ">=0.0.0", "*":
		return "*"
	default:
		return semverConstraint
	}
}
