package ubuntu

import (
	"bufio"
	"context"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"path/filepath"
	"regexp"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/cvs"
	"a.yandex-team.ru/security/yadi/libs/osreleases"
	"a.yandex-team.ru/security/yadi/snatcher/internal/gitutil"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/debversion"
)

const (
	trackerURI = "git://git.launchpad.net/ubuntu-cve-tracker"
	cveURL     = "https://ubuntu.com/security/%s"
	feedName   = "ubuntu-cve-tracker"
)

var (
	platforms = map[feed.Platform]string{
		feed.UbuntuPlatform: feed.UbuntuPlatform,
	}

	// order matters
	repoDirectories = []string{
		"retired",
		"active",
	}

	acceptableReleases = map[osreleases.Ubuntu]struct{}{
		osreleases.UbuntuUpstream: {},
		osreleases.UbuntuJammy:    {},
		osreleases.UbuntuImpish:   {},
		osreleases.UbuntuFocal:    {},
		osreleases.UbuntuBionic:   {},
		osreleases.UbuntuXenial:   {},
	}

	stableReleases = []osreleases.Ubuntu{
		osreleases.UbuntuJammy,
		osreleases.UbuntuFocal,
		osreleases.UbuntuBionic,
	}

	oldReleases = []osreleases.Ubuntu{
		osreleases.UbuntuHirsute,
		osreleases.UbuntuGroovy,
		osreleases.UbuntuEoan,
		osreleases.UbuntuDisco,
		osreleases.UbuntuCosmic,
		osreleases.UbuntuArtful,
		osreleases.UbuntuZesty,
		osreleases.UbuntuYakkety,
		osreleases.UbuntuWily,
		osreleases.UbuntuVivid,
		osreleases.UbuntuUtopic,
		osreleases.UbuntuTrusty,
		osreleases.UbuntuSaucy,
		osreleases.UbuntuRaring,
		osreleases.UbuntuQuantal,
		osreleases.UbuntuMaverick,
		osreleases.UbuntuLucid,
	}
	affectsCaptureRegexp      = regexp.MustCompile(`(?P<release>.*)_(?P<package>.*): (?P<status>[^\s]*)( \(+(?P<note>[^()]*)\)+)?`)
	affectsCaptureRegexpNames = affectsCaptureRegexp.SubexpNames()
)

type (
	Opts struct {
		TmpDir string
	}

	Feed struct {
		tmpDir string
	}
)

func NewFeed(opts Opts) (Feed, error) {
	return Feed{
		tmpDir: opts.TmpDir,
	}, nil
}

func (f Feed) Name() string {
	return feedName
}

func (f Feed) GetPlatformByAlias(alias string) (feed.Platform, error) {
	for p, a := range platforms {
		if a == alias {
			return p, nil
		}
	}
	return "", xerrors.New("not supported platform")
}

func (f Feed) Dump(ctx context.Context, _ feed.DumpingOpts) (feed.Result, error) {
	ubuntuRepo, err := gitutil.NewRepo(trackerURI, f.tmpDir)
	if err != nil {
		return nil, xerrors.Errorf("failed to create git repository: %w", err)
	}
	defer func() { _ = ubuntuRepo.Clean() }()

	repoPath, err := ubuntuRepo.CloneWithContext(ctx, 1)
	if err != nil {
		return nil, xerrors.Errorf("failed to clone git repository: %w", err)
	}

	eol := buildEol()
	vulns := map[feed.VulnID]feed.Vulnerability{
		eol.ID: eol,
	}

	for _, dirName := range repoDirectories {
		if err := processDirectory(filepath.Join(repoPath, dirName), vulns); err != nil {
			return nil, err
		}
	}

	return feed.Result{feed.UbuntuPlatform: vulns}, nil
}

func processDirectory(path string, vulns map[feed.VulnID]feed.Vulnerability) error {
	files, err := ioutil.ReadDir(path)
	if err != nil {
		return err
	}

	for _, f := range files {
		if f.IsDir() {
			continue
		}

		if !strings.HasPrefix(f.Name(), "CVE-") {
			continue
		}

		cveFilePath := filepath.Join(path, f.Name())
		if err := processCVEFile(cveFilePath, vulns); err != nil {
			return err
		}
	}
	return nil
}

func processCVEFile(path string, vulns map[feed.VulnID]feed.Vulnerability) error {
	simplelog.Info("process ubuntu CVE file", "path", path)
	file, err := os.Open(path)
	if err != nil {
		return err
	}

	defer func() { _ = file.Close() }()

	// Parse the vulnerability.
	vulnerabilities, err := parseUbuntuCVE(file)
	if err != nil {
		return xerrors.Errorf("failed to parse ubuntu CVE file %q: %w", path, err)
	}

	for vulnID, vuln := range vulnerabilities {
		vulns[vulnID] = vuln
	}
	return nil
}

func parseUbuntuCVE(fileContent io.Reader) (map[feed.VulnID]feed.Vulnerability, error) {
	var (
		vulnID          string
		cveID           string
		vulnDescription string
		vulnSourceURL   string
		vulnReferences  []feed.Reference
		vulnDisclosedAt int64
		vulnPkgs        = map[string]ubuntuVersions{}
		vulnCvssScore   float32

		readingDescription = false
		readingReferences  = false
	)

	scanner := bufio.NewScanner(fileContent)
	for scanner.Scan() {
		line := scanner.Text()

		if strings.HasPrefix(line, "#") {
			continue
		}

		if strings.HasPrefix(line, "Candidate:") {
			name := strings.TrimSpace(line[10:])
			cveID = strings.TrimSpace(name)
			vulnID = cveID
			vulnSourceURL = fmt.Sprintf(cveURL, cveID)
			continue
		}

		if strings.HasPrefix(line, "PublicDate:") {
			publicData := strings.TrimSpace(strings.TrimSpace(line[12:]))
			parsed, err := time.Parse("2006-01-02 15:04:05 MST", publicData)
			if err == nil {
				vulnDisclosedAt = parsed.Unix()
			}
			continue
		}

		if strings.HasPrefix(line, "Priority:") {
			priority := strings.TrimSpace(line[9:])
			// Skip any additional info
			if idx := strings.Index(priority, " "); idx > 0 {
				priority = priority[:idx]
			}
			vulnCvssScore = cvssScoreFromPriority(priority)
			continue
		}

		if strings.HasPrefix(line, "Description:") {
			if len(line) > 13 {
				vulnDescription = strings.TrimSpace(line[13:])
			}
			readingDescription = true
			continue
		}

		if readingDescription {
			if strings.HasPrefix(line, " ") {
				vulnDescription += " " + strings.TrimSpace(line)
				continue
			} else {
				vulnDescription = strings.TrimSpace(vulnDescription)
				readingDescription = false
			}
		}

		if strings.HasPrefix(line, "References:") {
			readingReferences = true
			continue
		}

		if readingReferences {
			if strings.HasPrefix(line, " ") {
				vulnReferences = append(vulnReferences, feed.Reference{
					URL: strings.TrimSpace(line),
				})
				continue
			} else {
				readingReferences = false
			}
		}

		affectsCaptureArr := affectsCaptureRegexp.FindAllStringSubmatch(line, -1)
		if len(affectsCaptureArr) > 0 {
			affectsCapture := affectsCaptureArr[0]

			md := map[string]string{}
			for i, n := range affectsCapture {
				md[affectsCaptureRegexpNames[i]] = strings.TrimSpace(n)
			}

			// Ignore Linux kernels.
			if strings.HasPrefix(md["package"], "linux") {
				continue
			}

			// Only consider the package if its status is needed, active, deferred, not-affected or
			// released. Ignore DNE (package does not exist), needs-triage, ignored, pending.
			if md["status"] == "needed" || md["status"] == "active" || md["status"] == "deferred" || md["status"] == "released" {
				if strings.Contains(md["release"], "/") {
					// Ignore any ESM releases
					continue
				}

				release := osreleases.UbuntuFromString(md["release"])
				if _, isAcceptable := acceptableReleases[release]; !isAcceptable {
					continue
				}

				var version string
				switch {
				case md["status"] == "released":
					if md["note"] == "" {
						continue
					}

					if _, err := debversion.NewVersion(md["note"]); err == nil {
						version = "<" + md["note"]
					} else {
						simplelog.Error("could not parse package version. skipping", "line", line)
						continue
					}
				case release == osreleases.UbuntuUpstream:
					continue
				default:
					version = "*"
				}

				uVersion := ubuntuVersion{
					Release: release,
					Vesion:  version,
				}

				vulnPkgs[md["package"]] = append(vulnPkgs[md["package"]], uVersion)
			}
		}
	}

	if vulnCvssScore < cvs.LowSeverity {
		return nil, nil
	}

	if vulnDescription == "" {
		vulnDescription = "(no description)"
	}

	vulnerabilities := map[feed.VulnID]feed.Vulnerability{}
	for pkgName, versions := range vulnPkgs {
		vulnVersions := versions.String()
		if vulnVersions == "" {
			continue
		}

		vulnID := fmt.Sprintf("%s-%s", pkgName, strings.TrimPrefix(vulnID, "CVE-"))
		yadiID := fmt.Sprintf("yadi-%s-%s", feed.UbuntuPlatform, vulnID)
		vulnReferences := append(vulnReferences[:], feed.Reference{
			Title: "Ubuntu CVE Tracker",
			URL:   vulnSourceURL,
		})

		vulnerabilities[vulnID] = feed.Vulnerability{
			ID:                 strings.ToUpper(vulnID),
			YadiID:             strings.ToUpper(yadiID),
			SrcType:            feedName,
			Title:              fmt.Sprintf("%s (%s)", pkgName, cveID),
			Description:        vulnDescription,
			DisclosedAt:        vulnDisclosedAt,
			References:         vulnReferences,
			CvssScore:          vulnCvssScore,
			Package:            pkgName,
			Language:           feed.UbuntuPlatform,
			VulnerableVersions: versions.String(),
		}
	}
	return vulnerabilities, nil
}

func buildEol() feed.Vulnerability {
	description := `## Overview
Using a obsolete Ubuntu release doesn't guarantee its security and stability.
You should upgrade to the nearest stable release:
`
	for _, release := range stableReleases {
		description += fmt.Sprintf("  * %s\n", strings.Title(release.String()))
	}

	vulnerableVersions := make(ubuntuVersions, len(oldReleases))
	for i, release := range oldReleases {
		vulnerableVersions[i] = ubuntuVersion{
			Release: release,
			Vesion:  "*",
		}
	}

	return feed.Vulnerability{
		Title:              "Obsolete Ubuntu Release",
		Package:            "ubuntu",
		ID:                 "OBSOLETE",
		DisclosedAt:        time.Now().Unix(),
		YadiID:             strings.ToUpper(fmt.Sprintf("yadi-%s-obsolete", feed.UbuntuPlatform)),
		SrcType:            feedName,
		Description:        description,
		RichDescription:    true,
		Language:           feed.UbuntuPlatform,
		VulnerableVersions: vulnerableVersions.String(),
		CvssScore:          9.9,
		References: []feed.Reference{
			{
				Title: "Ubuntu Releases",
				URL:   "https://wiki.ubuntu.com/Releases",
			},
		},
	}
}

func cvssScoreFromPriority(priority string) float32 {
	switch priority {
	case "untriaged", "negligible":
		return cvs.NoneSeverity
	case "low":
		return cvs.LowSeverity
	case "medium":
		return cvs.MediumSeverity
	case "high":
		return cvs.HighSeverity
	case "critical":
		return cvs.CriticalSeverity
	default:
		// TODO(buglloc): logging
		return cvs.NoneSeverity
	}
}
