package feed

import (
	"fmt"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/yadi/libs/cvs"
)

type (
	Reference struct {
		Title string `json:"title"`
		URL   string `json:"url"`
	}

	Vulnerability struct {
		References         []Reference `json:"references"`
		ID                 string      `json:"id"`
		SrcType            string      `json:"src_type"`
		YadiID             string      `json:"yadi_id" `
		Title              string      `json:"title"`
		Package            string      `json:"package"`
		Language           string      `json:"language"`
		VulnerableVersions string      `json:"vulnerable_versions"`
		Description        string      `json:"description"`
		PatchedVersions    string      `json:"patched_versions"`
		DisclosedAt        int64       `json:"disclosed"`
		CvssScore          float32     `json:"cvss_score"`
		PatchExists        bool        `json:"patch_exists"`
		RichDescription    bool        `json:"rich_description"` // true if description is not just a raw text (html/markdown/...)
	}
)

var ErrEmptyDescAndRefs = xerrors.New("the description and references cannot be empty at the same time")

// Compare returns true if general options of original Vulnerability did not changed
func (v Vulnerability) Compare(w Vulnerability) bool {
	return v.SrcType == w.SrcType &&
		v.ID == w.ID &&
		v.Language == w.Language &&
		v.Package == w.Package &&
		cvs.RoundBySeverity(v.CvssScore) == cvs.RoundBySeverity(w.CvssScore)
}

// StrictCompare it's like Compare, but stricter
func (v Vulnerability) StrictCompare(w Vulnerability) bool {
	return v.SrcType == w.SrcType &&
		v.ID == w.ID &&
		v.Language == w.Language &&
		v.Package == w.Package &&
		v.CvssScore == w.CvssScore &&
		v.VulnerableVersions == w.VulnerableVersions // && ... can be updated
}

// CheckVersionExpansion returns false in some strange cases when vulnerable versions expand to "*" from specific one.
// Returns true if all is OK
// This method is not commutative
func (v Vulnerability) CheckVersionExpansion(new Vulnerability) bool {
	return v.VulnerableVersions == "*" || new.VulnerableVersions != "*"
}

// Validate returns error if some of general options does not valid
func (v Vulnerability) Validate() error {
	// check for empty fields
	if v.SrcType == "" {
		return xerrors.New("empty src_type")
	}
	if v.ID == "" {
		return xerrors.New("empty id")
	}
	if v.Title == "" {
		return xerrors.New("empty title")
	}
	if v.Package == "" {
		return xerrors.New("empty package")
	}
	if v.Language == "" {
		return xerrors.New("empty language")
	}
	if v.VulnerableVersions == "" {
		return xerrors.New("empty vulnerable_versions")
	}

	if v.CvssScore < 0 || v.CvssScore > 10 {
		return xerrors.New("invalid cvss_score: must be between 0 and 10")
	}

	if _, ok := Platforms[v.Language]; !ok {
		return xerrors.Errorf("invalid language: %s", v.Language)
	}

	// this check must be the last one, because caller checks error
	// TODO(melkikh): fixme
	if v.Description == "" && len(v.References) == 0 {
		return ErrEmptyDescAndRefs
	}
	return nil
}

// StrictValidate returns error if some of general options does not valid. It is stricter than Validate
func (v Vulnerability) StrictValidate() error {
	if err := v.Validate(); err != nil {
		return err
	}

	if !(v.CvssScore == cvs.NoneSeverity ||
		v.CvssScore == cvs.LowSeverity ||
		v.CvssScore == cvs.MediumSeverity ||
		v.CvssScore == cvs.HighSeverity ||
		v.CvssScore == cvs.CriticalSeverity) {
		return xerrors.New("invalid cvss_score: must be rounded by severity")
	}

	return nil
}

// Adjusts tries to put Vulnerability in good state to pass the Validate() method
func (v *Vulnerability) Adjust() error {
	v.CvssScore = cvs.RoundBySeverity(v.CvssScore)
	// TODO(melkikh): think about something else
	return nil
}

func (v Vulnerability) String() string {
	return fmt.Sprintf("%s:%s", v.SrcType, v.ID)
}
