package wiki

import (
	"fmt"
	"regexp"
	"strconv"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/libs/cvs"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed"
)

type (
	cellKey = string

	TableCell struct {
		Sort        string      `json:"sort,omitempty"`
		Raw         interface{} `json:"raw"`
		Transformed string      `json:"transformed,omitempty"`
		RowID       string      `json:"row_id"`
		Key         cellKey     `json:"__key__"`
		Status      string      `json:"status,omitempty"`
		URL         string      `json:"url,omitempty"`
	}
	TableRow []TableCell
	Page     struct {
		Error struct {
			DebugMessage string   `json:"debug_message"`
			ErrorCode    string   `json:"error_code"`
			Level        string   `json:"level"`
			Message      []string `json:"message"`
		} `json:"error,omitempty"`
		Data struct {
			Rows []TableRow `json:"rows"`
		} `json:"data,omitempty"`
	}
)

var (
	YadiIDPkgBadChars  = regexp.MustCompile("[^a-zA-Z0-9]+")
	BadRawInterfaceErr = "Bad type of cell raw value: %i"
)

const (
	wikiKeyTitle              = "100"
	wikiKeyLanguage           = "101"
	wikiKeyIssue              = "102"
	wikiKeyPackageName        = "103"
	wikiKeyVulnerableVersions = "104"
	wikiKeyPatchedVersions    = "105"
	wikiKeyDescription        = "106"
	wikiKeyDate               = "107"
	wikiKeySeverity           = "108"
)

func (f Feed) NewVuln(row TableRow, rowNumber uint64) (*feed.Vulnerability, error) {
	v := new(feed.Vulnerability)
	var err error
	v.SrcType = f.Name()

	for _, cell := range row {
		switch cell.Key {
		case wikiKeyTitle:
			v.Title, err = ifaceToString(cell.Raw)
			if err != nil {
				return nil, err
			}
		case wikiKeyLanguage:
			alias, err := ifaceSliceToString(cell.Raw)
			if err != nil {
				return nil, err
			}

			lang, err := f.GetPlatformByAlias(alias)
			if err != nil {
				return nil, err
			}
			v.Language = lang
		case wikiKeyIssue:
			ref, err := ifaceToString(cell.Raw)
			if err != nil {
				return nil, err
			}
			v.References = []feed.Reference{
				{
					Title: "Yandex",
					URL:   trackerIssueURI + ref,
				},
			}
		case wikiKeyPackageName:
			v.Package, err = ifaceToString(cell.Raw)
			if err != nil {
				return nil, err
			}

		case wikiKeyVulnerableVersions:
			v.VulnerableVersions, err = ifaceToString(cell.Raw)
			if err != nil {
				return nil, err
			}
		case wikiKeyPatchedVersions:
			v.PatchedVersions, err = ifaceToString(cell.Raw)
			if err != nil {
				return nil, err
			}
		case wikiKeyDescription:
			var text strings.Builder
			description, err := ifaceToString(cell.Raw)
			if err != nil {
				return nil, err
			}
			text.WriteString("## Описание \n")
			text.WriteString(description)
			v.Description = text.String()
			v.RichDescription = true
		case wikiKeyDate:
			dateStr, err := ifaceToString(cell.Raw)
			if err != nil {
				return nil, err
			}
			date, err := time.Parse("2006-01-02", dateStr)
			if err != nil {
				return nil, err
			}
			v.DisclosedAt = date.Unix()
		case wikiKeySeverity:
			severity, err := ifaceSliceToString(cell.Raw)
			if err != nil {
				return nil, err
			}
			v.CvssScore, err = cvs.FromSeverity(severity)
			if err != nil {
				return nil, err
			}

		default:
			continue
		}
	}
	v.YadiID = "YADI" + "-" + strings.ToUpper(v.Language) + "-" + YadiIDSuffix(v.Package, rowNumber)
	v.ID = v.YadiID

	return v, nil
}

func YadiIDSuffix(pkgName string, rowNumber uint64) string {
	normalizedPkgName := YadiIDPkgBadChars.ReplaceAllLiteralString(pkgName, "-") // fixme
	normalizedPkgName = strings.Trim(normalizedPkgName, "-")
	if len(normalizedPkgName) < 2 {
		simplelog.Error(fmt.Sprintf("Can't normalize package name: too short [%d]", rowNumber))
	}

	return strings.ToUpper(normalizedPkgName + "-" + strconv.FormatUint(rowNumber, 10))
}

func ifaceToString(i interface{}) (string, error) {
	switch r := i.(type) {
	case string:
		return r, nil
	default:
		return "", xerrors.Errorf(BadRawInterfaceErr, r)
	}
}

func ifaceSliceToString(i interface{}) (string, error) {
	switch r := i.(type) {
	case []interface{}:
		return r[0].(string), nil
	default:
		return "", xerrors.Errorf(BadRawInterfaceErr, r)
	}
}
