package wiki

import (
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/retry"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/libs/go/yahttp"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed"
)

const (
	wikiDBURI       = "https://wiki-api.yandex-team.ru/_api/frontend/product-security/yadi/vulns/db/.grid?format=json"
	trackerIssueURI = "https://st.yandex-team.ru/"
)

type (
	Opts struct {
		OAuthToken string
	}

	Feed struct {
		token   string
		fetcher *http.Client
	}
)

var (
	platforms = map[feed.Platform]string{
		feed.NodeJSPlatform: "Node.js",
		feed.PythonPlatform: "Python",
		feed.JavaPlatform:   "Java",
	}
)

func NewFeed(opts Opts) (Feed, error) {
	return Feed{
		token: opts.OAuthToken,
		fetcher: yahttp.NewClient(yahttp.Config{
			RedirectPolicy: yahttp.RedirectFollow,
			Timeout:        5 * time.Second,
		}),
	}, nil
}

func (f Feed) Name() string {
	return "wiki"
}

func (f Feed) GetPlatformByAlias(alias string) (feed.Platform, error) {
	for p, a := range platforms {
		if a == alias {
			return p, nil
		}
	}
	return "", xerrors.New("not supported platform")
}

func (f Feed) Dump(ctx context.Context, opts feed.DumpingOpts) (feed.Result, error) {
	request, err := http.NewRequestWithContext(ctx, "GET", wikiDBURI, nil)
	if err != nil {
		simplelog.Error("Failed to create request", "err", err)
		return feed.Result{}, err
	}

	request.Header.Add("Authorization", fmt.Sprintf("OAuth %s", f.token))

	retrier := retry.New(
		retry.WithAttempts(5),
	)

	var r *http.Response
	err = retrier.Try(ctx, func(ctx context.Context) error {
		r, err = f.fetcher.Do(request)
		if err != nil {
			simplelog.Error("Failed to fetch feed", "err", err)
			return err
		}

		if r.StatusCode != http.StatusOK {
			var bodyString string
			bodyBytes, err := ioutil.ReadAll(r.Body)
			if err == nil {
				bodyString = string(bodyBytes)
			}
			simplelog.Warn("Bad API response", "response", fmt.Sprintf("%d: %s", r.StatusCode, bodyString))
			return xerrors.New("feed not available now")
		}
		return nil
	})
	if err != nil {
		simplelog.Error("Failed to fetch wiki page", "err", err)
		return feed.Result{}, err
	}

	page := new(Page)
	defer yahttp.GracefulClose(r.Body)

	err = json.NewDecoder(r.Body).Decode(&page)
	if err != nil {
		simplelog.Error("Failed to decode feed JSON", "err", err)
		return feed.Result{}, err
	}

	result := feed.Result{}
	for platform := range platforms {
		result[platform] = make(map[feed.VulnID]feed.Vulnerability)
	}

	for n, row := range page.Data.Rows {
		vuln, err := f.NewVuln(row, uint64(n))
		if err != nil {
			simplelog.Error("Failed to create vulnerability from table row", "err", err)
			continue
		}
		result[vuln.Language][vuln.ID] = *vuln
	}

	return result, nil
}
