package manifestor

import (
	"fmt"
	"time"
)

type Manifest struct {
	Timestamp int64             `json:"timestamp"`
	Files     []string          `json:"files"` //
	Hashes    map[string]string `json:"hashsums"`
}

func Empty() *Manifest {
	return &Manifest{
		Files:     make([]string, 0),
		Hashes:    make(map[string]string),
		Timestamp: time.Now().Unix(),
	}
}

func (m *Manifest) Update(ts int64, path, hash string) {
	var fileExists bool
	for i := range m.Files {
		if m.Files[i] == path {
			fileExists = true
		}
	}
	if !fileExists {
		m.Files = append(m.Files, path)
	}

	m.Hashes[path] = hash
	m.Timestamp = ts
}

func (m *Manifest) Validate() error {
	if m.Timestamp <= 0 {
		return fmt.Errorf("invalid timestamp: %d", m.Timestamp)
	}

	var hashKeys []string
	for filename := range m.Hashes {
		hashKeys = append(hashKeys, filename)
	}

	if !sameSlicesWithoutOrder(hashKeys, m.Files) {
		return fmt.Errorf("files and hashsums have different filenames: hashes = %v, files = %v", hashKeys, m.Files)
	}

	return nil
}

func sameSlicesWithoutOrder(x, y []string) bool {
	if len(x) != len(y) {
		return false
	}
	diff := make(map[string]int, len(x))
	for _, _x := range x {
		diff[_x]++
	}
	for _, _y := range y {
		if _, ok := diff[_y]; !ok {
			return false
		}
		diff[_y] -= 1
		if diff[_y] == 0 {
			delete(diff, _y)
		}
	}
	return len(diff) == 0
}
