/*
Package unifiedrange parses "unified range" versions (https://github.com/snyk/unified-range) to Semantic Versions (semver.org)

Example for unified range versions:
https://github.com/snyk/unified-range/blob/master/tests/test_data/unified-ranges.txt
[,1.0.1)
[,1.0.1), [1.1,1.1.10), [1.2,1.2.8), [1.3,1.3.2), [1.4,1.4.14), [1.5,1.5.3), [1.6,1.6.2), [1.7,1.7.1), [1.8,1.8.1)
[,1.0.1.14)

*/
package unifiedrange

import (
	"bytes"
	"fmt"
	"regexp"
	"strconv"
	"strings"
	"text/template"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/libs/go/semver"
)

var (
	UnifiedRe     = regexp.MustCompile(`(?P<lborder>[\[\(]{1})(?P<left>(?P<lmajor>[\d\w-]+)?\.?(?P<lminor>[\d\w-]+)?\.?(?P<lpatch>[\d\w-]+)?[\.\d\w]*|[0]|)\s*(?P<sep>,?)\s*(?P<right>(?P<rmajor>[\d\w-]+)?\.?(?P<rminor>[\d\w-]+)?\.?(?P<rpatch>[\d\w-]+)?[\.\d\w]*|[0]|)(?P<rborder>[\]\)]{1})`)
	SemverTmplStr = `{{ if .Left }}{{ if .LeftOpen }}>{{ else }}>={{ end }}{{ .LeftVer }}{{ if .Right }} {{ else }}{{end}}{{ end }}{{ if .Right }}{{ if .RightOpen }}<{{ else }}<={{ end }}{{ .RightVer }}{{ end }}`
	SevmerTmpl    = template.Must(template.New("semverRange").Parse(SemverTmplStr))
)

type Range struct {
	LeftVer   string
	RightVer  string
	Left      bool
	LeftOpen  bool
	Right     bool
	RightOpen bool
}

func ToSemverSeries(ufrange string) (result string, err error) {
	var resultArr []string
	matches := UnifiedRe.FindAllStringSubmatch(ufrange, -1)
	for _, match := range matches {
		sv, err := toSemver(match)
		if err != nil {
			return "", err
		}
		resultArr = append(resultArr, sv)
	}
	return strings.Join(resultArr, " || "), nil
}

func ToSemverConstraint(ufrange string) (result string, err error) {
	match := UnifiedRe.FindStringSubmatch(ufrange)
	return toSemver(match)
}

func IsUnifiedRange(version string) bool {
	return UnifiedRe.Match([]byte(version))
}

func toSemver(match []string) (result string, err error) {

	regexpResult := make(map[string]string)
	semverRange := Range{}

	for i, name := range UnifiedRe.SubexpNames() {
		if i != 0 && name != "" {
			regexpResult[name] = match[i]
		}
	}

	if regexpResult["sep"] != "," {
		result = fmt.Sprintf("==%s", joinMajorMinorPatch(regexpResult["lmajor"], regexpResult["lminor"], regexpResult["lpatch"]))
		_, err := semver.NewConstraint(result)
		return result, err
	}

	if regexpResult["lborder"] == "(" {
		semverRange.LeftOpen = true
	} else if regexpResult["lborder"] != "[" {
		return result, xerrors.New("Wrong left endpoint")
	}

	if regexpResult["rborder"] == ")" {
		semverRange.RightOpen = true
	} else if regexpResult["rborder"] != "]" {
		return result, xerrors.New("Wrong right endpoint")
	}

	if regexpResult["left"] != "" {
		semverRange.Left = true
		semverRange.LeftVer = joinMajorMinorPatch(regexpResult["lmajor"], regexpResult["lminor"], regexpResult["lpatch"])
	}

	if regexpResult["right"] != "" {
		semverRange.Right = true
		semverRange.RightVer = joinMajorMinorPatch(regexpResult["rmajor"], regexpResult["rminor"], regexpResult["rpatch"])
	}

	// YADI-70
	// [0, ) or (0, ) -> *
	if semverRange.Left && !semverRange.Right &&
		cleanNumber(regexpResult["lmajor"]) == 0 &&
		cleanNumber(regexpResult["lminor"]) == 0 &&
		cleanNumber(regexpResult["lpatch"]) == 0 {
		return "*", nil
	}

	// (, 0) or (, 0] -> <0.0.0
	if !semverRange.Left && semverRange.Right &&
		cleanNumber(regexpResult["rmajor"]) == 0 &&
		cleanNumber(regexpResult["rminor"]) == 0 &&
		cleanNumber(regexpResult["rpatch"]) == 0 {
		return "<0.0.0", nil
	}

	var tpl bytes.Buffer
	if err := SevmerTmpl.Execute(&tpl, semverRange); err != nil {
		return result, err
	}

	result = tpl.String()
	_, err = semver.NewConstraint(result)
	if err != nil {
		return result, err
	}

	return
}

func joinMajorMinorPatch(major string, minor string, patch string) string {
	return fmt.Sprintf("%d.%d.%d", cleanNumber(major), cleanNumber(minor), cleanNumber(patch))
}

func cleanNumber(num string) int {
	if r, err := strconv.Atoi(num); err == nil {
		return r
	}

	var r int
	for _, char := range num {
		if '0' <= char && char <= '9' {
			r = r*10 + (int(char) - '0')
		} else {
			break
		}

	}
	return r
}
