package main

import (
	"context"
	"fmt"
	"net/url"
	"os"
	"os/signal"
	"syscall"
	"time"

	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/web/internal/config"
	"a.yandex-team.ru/security/yadi/web/internal/infra"
	"a.yandex-team.ru/security/yadi/web/internal/server"
)

const (
	shutdownDeadline = time.Second * 30
)

func nonEmpty(sources ...string) string {
	for _, str := range sources {
		if str != "" {
			return str
		}
	}
	panic("empty strings")
}

func urlOrPanic(raw string) url.URL {
	res, err := url.Parse(raw)
	if err != nil {
		panic(fmt.Errorf("failed to parse URL %s: %w", raw, err))
	}
	return *res
}

func newConfig() config.Config {
	var cfg config.Config
	switch os.Getenv("YADI_ENV") {
	case "dev":
		simplelog.SetLevel(simplelog.DebugLevel)
		cfg = config.Config{
			Debug:           true,
			AdminUsers:      []string{"buglloc"},
			AdminGroups:     []string{"yandex_mnt_security_1327"},
			WebURL:          nonEmpty(os.Getenv("YADI_WEB_URL"), "http://localhost:3001"),
			YdbEndpoint:     "ydb-ru.yandex.net:2135",
			YdbDatabase:     nonEmpty(os.Getenv("YDB_DATABASE"), "/ru/home/robot-yadi/mydb"),
			YdbPath:         nonEmpty(os.Getenv("YDB_PATH"), "dev"),
			HTTPPort:        3001,
			CsrfKey:         "some-csrf-key-1234",
			VulnDBPath:      urlOrPanic(nonEmpty(os.Getenv("YADI_DB_PATH"), "https://yaditestdb.s3.mdst.yandex.net/")),
			ManifestPath:    urlOrPanic(nonEmpty(os.Getenv("MANIFEST_PATH"), "https://yaditestdb.s3.mdst.yandex.net/manifest.json")),
			WatcherCronSpec: nonEmpty(os.Getenv("FEED_UPDATE_CRON_SPEC"), "0 * * * * *"),
		}
	case "test":
		simplelog.SetLevel(simplelog.DebugLevel)
		cfg = config.Config{
			Debug:           true,
			AdminUsers:      []string{"buglloc"},
			AdminGroups:     []string{"yandex_mnt_security_1327"},
			WebURL:          nonEmpty(os.Getenv("YADI_WEB_URL"), "https://yadi-test.sec.yandex-team.ru"),
			YdbEndpoint:     "ydb-ru.yandex.net:2135",
			YdbDatabase:     nonEmpty(os.Getenv("YDB_DATABASE"), "/ru/home/robot-yadi/mydb"),
			YdbPath:         nonEmpty(os.Getenv("YDB_PATH"), "dev"),
			HTTPPort:        80,
			CsrfKey:         os.Getenv("CSRF_KEY"),
			VulnDBPath:      urlOrPanic(nonEmpty(os.Getenv("YADI_DB_PATH"), "https://yaditestdb.s3.mdst.yandex.net/")),
			ManifestPath:    urlOrPanic(nonEmpty(os.Getenv("MANIFEST_PATH"), "https://yaditestdb.s3.mdst.yandex.net/manifest.json")),
			WatcherCronSpec: nonEmpty(os.Getenv("FEED_UPDATE_CRON_SPEC"), "0 */10 * * * *"),
		}
	case "prod":
		fallthrough
	default:
		cfg = config.Config{
			Debug:           false,
			AdminUsers:      []string{"buglloc"},
			AdminGroups:     []string{"yandex_mnt_security_1327"},
			WebURL:          nonEmpty(os.Getenv("YADI_WEB_URL"), "https://yadi.yandex-team.ru"),
			YdbEndpoint:     "ydb-ru.yandex.net:2135",
			YdbDatabase:     nonEmpty(os.Getenv("YDB_DATABASE"), "/ru/home/robot-yadi/mydb"),
			YdbPath:         nonEmpty(os.Getenv("YDB_PATH"), "prod"),
			HTTPPort:        80,
			CsrfKey:         os.Getenv("CSRF_KEY"),
			VulnDBPath:      urlOrPanic(nonEmpty(os.Getenv("YADI_DB_PATH"), "https://yadidb.s3.mds.yandex.net/")),
			ManifestPath:    urlOrPanic(nonEmpty(os.Getenv("MANIFEST_PATH"), "https://yadidb.s3.mds.yandex.net/manifest.json")),
			WatcherCronSpec: nonEmpty(os.Getenv("FEED_UPDATE_CRON_SPEC"), "0 * * * * *"),
		}
	}

	return cfg
}

func main() {
	_ = maxprocs.AdjustAuto()

	cfg := newConfig()
	srv, err := server.New(infra.New(cfg))
	if err != nil {
		panic(err)
	}

	stopChan := make(chan os.Signal, 1)
	signal.Notify(stopChan, syscall.SIGINT, syscall.SIGTERM)

	errChan := make(chan error, 1)
	go func() {
		simplelog.Info("starting server")
		if err := srv.ListenAndServe(cfg.HTTPPort); err != nil {
			errChan <- err
		}
	}()

	select {
	case <-stopChan:
		ctx, cancel := context.WithTimeout(context.Background(), shutdownDeadline)
		defer cancel()

		if err := srv.Shutdown(ctx); err != nil {
			simplelog.Error("failed to shutdown yaudit server", "err", err)
		}
	case err := <-errChan:
		simplelog.Error("unexpected HTTP server error", "err", err)
	}
}
