package confirmator

import (
	"net/http"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/security/yadi/web/internal/echoutils"
	"a.yandex-team.ru/security/yadi/web/internal/infra"
)

type Controller struct {
	*infra.Infra
}

func (c *Controller) BuildRoute(g *echo.Group) error {
	g.GET("/confirm/:service/:name", c.confirmGet)
	g.POST("/confirm/:service/:name", c.confirmPost)
	return nil
}

func (c *Controller) confirmGet(e echo.Context) error {
	user, err := c.getUser(e)
	if err != nil {
		return echoutils.PageError(e, err)
	}

	service := e.Param("service")
	name := e.Param("name")
	project, err := c.DB.LookupProjectInfo(service, name)
	if err != nil {
		return echoutils.PageError(e, err)
	}

	var yandexUID string
	if cYandexUID, err := e.Cookie("yandexuid"); err == nil {
		yandexUID = cYandexUID.Value
	}

	sk, err := c.YaCsrf.Generate(user.ID, yandexUID)
	if err != nil {
		return echoutils.PageError(e, err)
	}

	return e.Render(http.StatusOK, "project-confirm.html", echo.Map{
		"service": service,
		"name":    name,
		"active":  project.Active,
		"sk":      sk,
	})
}

func (c *Controller) confirmPost(e echo.Context) error {
	user, err := c.getUser(e)
	if err != nil {
		return echoutils.PageError(e, err)
	}

	service := e.Param("service")
	name := e.Param("name")
	project, err := c.DB.LookupProjectInfo(service, name)
	if err != nil {
		return echoutils.PageError(e, err)
	}

	var yandexUID string
	if cYandexUID, err := e.Cookie("yandexuid"); err == nil {
		yandexUID = cYandexUID.Value
	}

	sk := e.FormValue("sk")
	if err = c.YaCsrf.Validate(sk, user.ID, yandexUID); err != nil {
		return echoutils.PageError(e, err)
	}

	project.Active = true
	project.CreatedBy = user.ID
	if err = c.DB.ActivateProject(project); err != nil {
		return echoutils.PageError(e, err)
	}

	return e.Render(http.StatusOK, "project-confirmed.html", echo.Map{
		"service": service,
		"name":    name,
	})
}

func (c *Controller) getUser(e echo.Context) (*blackbox.User, error) {
	sessID, err := e.Cookie("Session_id")
	if err != nil {
		return nil, xerrors.New("no session")
	}

	rsp, err := c.BlackBox.SessionID(
		e.Request().Context(),
		blackbox.SessionIDRequest{
			SessionID: sessID.Value,
			UserIP:    e.RealIP(),
			Host:      e.Request().Host,
		})

	if err != nil {
		return nil, xerrors.Errorf("failed to check user session: %w", err)
	}

	return &rsp.User, nil
}
