package moderator

import (
	"strconv"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/yadi/snatcher/pkg/feed"
	"a.yandex-team.ru/security/yadi/web/internal/db"
	"a.yandex-team.ru/security/yadi/web/internal/echoutils"
	"a.yandex-team.ru/security/yadi/web/internal/infra"
)

const YdbMaxRows = 50 // Rows per query. Max: 1000

type (
	Controller struct {
		*infra.Infra
	}
)

func (c *Controller) BuildRoute(g *echo.Group) error {
	g.GET("/csrf", c.generateYaCsrf)

	g.GET("/vuln/:yaid", c.getYaVuln)
	g.GET("/vuln/:src/:id", c.getVuln)

	g.GET("/list/new", c.listNew)
	g.GET("/list/changed", c.listChanged)
	g.GET("/list/all", c.listAll)

	g.POST("/approve", c.approve, c.checkAdminRequest)
	g.POST("/change", c.change, c.checkAdminRequest)
	g.POST("/reject", c.reject, c.checkAdminRequest)
	return nil
}

func (c *Controller) generateYaCsrf(e echo.Context) error {
	user, err := c.getCookieUser(e)
	if err != nil {
		return echoutils.APIError(e, err)
	}

	sk, err := c.YaCsrf.Generate(user.ID, c.getYandexUID(e))
	if err != nil {
		return echoutils.APIError(e, err)
	}

	return echoutils.APIOk(e, echo.Map{
		"sk": sk,
	})
}

func (c *Controller) getYaVuln(e echo.Context) error {
	result, err := c.DB.LookupOneFeedVuln(db.VulnLookupOpts{
		YaID: e.Param("yaid"),
	})
	if err != nil {
		if err == db.ErrNotFound {
			return echoutils.APINotFound(e)
		}
		return echoutils.APIError(e, err)
	}
	return echoutils.APIOk(e, result)
}

func (c *Controller) getVuln(e echo.Context) error {
	result, err := c.DB.LookupOneFeedVuln(db.VulnLookupOpts{
		SrcType: e.Param("src"),
		SrcID:   e.Param("id"),
	})
	if err != nil {
		if err == db.ErrNotFound {
			return echoutils.APINotFound(e)
		}
		return echoutils.APIError(e, err)
	}
	return echoutils.APIOk(e, result)
}

func (c *Controller) listNew(e echo.Context) error {
	// select vuln from "src" and "action" tables, that not included in "feed" table
	result, err := c.DB.LookupAddedVulns(db.FeedLookupOpts{
		LastKey: parseLastKey(e),
		Limit:   YdbMaxRows,
	})
	if err != nil {
		if err == db.ErrNotFound {
			return echoutils.APINotFound(e)
		}
		return echoutils.APIError(e, err)
	}
	response, err := newListNewResponse(e, result)
	if err != nil {
		return echoutils.APIError(e, err)
	}
	return echoutils.APIOk(e, response)
}

func (c *Controller) listChanged(e echo.Context) error {
	// select vulns from "feed" table with some field from "src" and "action" tables
	result, err := c.DB.LookupChangedVulns(db.FeedLookupOpts{
		LastKey: parseLastKey(e),
		Limit:   YdbMaxRows,
	})
	if err != nil {
		if err == db.ErrNotFound {
			return echoutils.APINotFound(e)
		}
		return echoutils.APIError(e, err)
	}
	response, err := newListChangedResponse(e, result)
	if err != nil {
		return echoutils.APIError(e, err)
	}
	return echoutils.APIOk(e, response)
}

func (c *Controller) listAll(e echo.Context) error {
	result, err := c.DB.LookupAllFeed(db.FeedLookupOpts{
		LastKey: parseLastKey(e),
		Limit:   YdbMaxRows,
	})
	if err != nil {
		if err == db.ErrNotFound {
			return echoutils.APINotFound(e)
		}
		return echoutils.APIError(e, err)
	}
	response, err := newListAllResponse(e, result)
	if err != nil {
		return echoutils.APIError(e, err)
	}
	return echoutils.APIOk(e, response)
}

func (c *Controller) approve(e echo.Context) error {
	reqVuln := new(feed.Vulnerability)
	if err := e.Bind(reqVuln); err != nil {
		return echoutils.APIError(e, err)
	}

	if err := c.DB.ApproveVuln(reqVuln); err != nil {
		return echoutils.APIError(e, xerrors.Errorf("failed to approve vuln %q: %w", reqVuln, err))
	}

	return echoutils.APIOk(e, "OK")
}

func (c *Controller) change(e echo.Context) error {
	validate := func(v *feed.Vulnerability) (err error) {
		if err = v.Adjust(); err != nil {
			return xerrors.Errorf("can not adjust %s vuln: %w", v.ID, err)
		}

		if err = v.StrictValidate(); err == nil {
			return nil
		}
		if xerrors.Is(err, feed.ErrEmptyDescAndRefs) {
			// there are no Refs and Description in this handler
			return nil
		}
		return err
	}

	vuln := new(feed.Vulnerability)
	if err := e.Bind(vuln); err != nil {
		return echoutils.APIError(e, err)
	}

	if err := validate(vuln); err != nil {
		return echoutils.APIError(e, err)
	}

	if err := c.DB.ChangeVuln(vuln); err != nil {
		return echoutils.APIError(e, err)
	}

	return echoutils.APIOk(e, "OK") // fixme
}

func (c *Controller) reject(e echo.Context) error {
	vuln := new(feed.Vulnerability)
	if err := e.Bind(vuln); err != nil {
		return echoutils.APIError(e, err)
	}
	err := c.DB.DeleteAction(vuln.SrcType, vuln.ID)
	if err != nil {
		return echoutils.APIError(
			e,
			xerrors.Errorf("failed to delete vulnerability %q (%s): %w", vuln, vuln.ID, err),
		)
	}
	return echoutils.APIOk(e, "OK")
}

func parseLastKey(e echo.Context) (key uint64) {
	gt := e.QueryParam("gt")
	if gt == "" {
		return
	} else if s, err := strconv.ParseUint(gt, 10, 64); err != nil {
		return
	} else {
		return s
	}
}
