package moderator

import (
	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/security/yadi/web/internal/echoutils"
)

const YaCsrfHdr = "X-CSRF-Token"

// TODO(buglloc): move into server-wide middleware
func (c *Controller) checkAdminRequest(next echo.HandlerFunc) echo.HandlerFunc {
	return func(e echo.Context) error {
		var user *blackbox.User
		var err error
		switch {
		case e.Request().Header.Get("Authorization") != "":
			user, err = c.getOAuthUser(e)
		default:
			token := e.Request().Header.Get(YaCsrfHdr)
			if token == "" {
				return echoutils.APIError(e, xerrors.Errorf("Invalid request. Need %s header", YaCsrfHdr))
			}

			user, err = c.getCookieUser(e)
			if err != nil {
				return echoutils.APIError(e, err)
			}

			err = c.YaCsrf.Validate(token, user.ID, c.getYandexUID(e))
		}

		if err != nil {
			return echoutils.APIError(e, err)
		}

		if !c.ACLer.CheckUser(user.Login) {
			return echoutils.APIError(
				e,
				xerrors.Errorf("Access Denied (user %q does not have required role)", user.Login),
			)
		}

		return next(e)
	}
}

func (c *Controller) getCookieUser(e echo.Context) (*blackbox.User, error) {
	sessID, err := e.Cookie("Session_id")
	if err != nil {
		return nil, xerrors.New("no session")
	}

	rsp, err := c.BlackBox.SessionID(
		e.Request().Context(),
		blackbox.SessionIDRequest{
			SessionID: sessID.Value,
			UserIP:    e.RealIP(),
			Host:      e.Request().Host,
		})

	if err != nil {
		return nil, xerrors.Errorf("failed to check user session: %w", err)
	}

	return &rsp.User, nil
}

func (c *Controller) getOAuthUser(e echo.Context) (*blackbox.User, error) {
	token := e.Request().Header.Get("Authorization")
	if len(token) <= 6 || token[:6] != "OAuth " {
		return nil, xerrors.New("failed to parse OAuth token")
	}

	rsp, err := c.BlackBox.OAuth(
		e.Request().Context(),
		blackbox.OAuthRequest{
			OAuthToken: token[6:],
			UserIP:     e.RealIP(),
			Scopes:     []string{"yadi:all"},
		})

	if err != nil {
		return nil, xerrors.Errorf("failed to check OAuth token: %w", err)
	}

	return &rsp.User, nil
}

func (c *Controller) getYandexUID(e echo.Context) string {
	var yandexUID string
	if cYandexUID, err := e.Cookie("yandexuid"); err == nil {
		yandexUID = cYandexUID.Value
	}
	return yandexUID
}
