package project

import (
	"fmt"
	"net/url"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/security/yadi/web/internal/db"
	"a.yandex-team.ru/security/yadi/web/internal/echoutils"
	"a.yandex-team.ru/security/yadi/web/internal/infra"
	"a.yandex-team.ru/security/yadi/web/internal/models"
	"a.yandex-team.ru/security/yadi/web/internal/validators"
)

type Controller struct {
	*infra.Infra
}

func (c *Controller) BuildRoute(g *echo.Group) error {
	g.POST("/new", c.projectNew)
	g.GET("/:service/:name", c.projectGet)
	g.POST("/:service/:name/sync", c.projectSync)
	g.GET("/:service/:name/dependencies", c.projectDeps)
	return nil
}

func (c *Controller) projectNew(e echo.Context) error {
	project := new(models.Project)
	if err := e.Bind(project); err != nil {
		return echoutils.APIError(e, err)
	}

	if err := validators.AbcService(project.Service); err != nil {
		return echoutils.APIError(e, err)
	}

	if err := validators.Owners(project.Owners); err != nil {
		return echoutils.APIError(e, err)
	}

	if err := validators.Severity(project.Severity); err != nil {
		return echoutils.APIError(e, err)
	}

	if err := validators.ProjectUniq(c.Infra, project.Service, project.Name); err != nil {
		return echoutils.APIError(e, err)
	}

	err := c.DB.CreateProject(project)
	if err != nil {
		return echoutils.APIError(e, err)
	}

	confirmURI := fmt.Sprintf(
		"https://%s/project/confirm/%s/%s/",
		e.Request().Host,
		url.PathEscape(project.Service),
		url.PathEscape(project.Name),
	)
	return echoutils.APIOk(e, echo.Map{
		"confirm_uri": confirmURI,
	})
}

func (c *Controller) projectGet(e echo.Context) error {
	if _, err := c.getUser(e); err != nil {
		return echoutils.APIError(e, err)
	}

	service := e.Param("service")
	name := e.Param("name")
	project, err := c.DB.LookupProjectInfo(service, name)
	if err != nil {
		if err != db.ErrNotFound {
			return echoutils.APIError(e, xerrors.Errorf("failed to get project %s/%s: %w", service, name, err))
		}
		return echoutils.APINotFound(e)
	}

	return echoutils.APIOk(e, project)
}

func (c *Controller) projectSync(e echo.Context) error {
	return echoutils.APIOk(e, nil)
	//TODO(buglloc): enable later
	//
	//if _, err := c.getUser(e); err != nil {
	//	return echoutils.APIError(e, err)
	//}
	//
	//// Ugly hack for old clients :(
	//e.Request().Header.Set(echo.HeaderContentType, echo.MIMEApplicationJSON)
	//
	//syncInfo := new(models.SyncInfo)
	//if err := e.Bind(syncInfo); err != nil {
	//	return echoutils.APIError(e, err)
	//}
	//
	//syncInfo.Service = e.Param("service")
	//syncInfo.Name = e.Param("name")
	//if err := validators.ProjectExists(c.Infra, syncInfo.Service, syncInfo.Name); err != nil {
	//	return echoutils.APIError(e, err)
	//}
	//
	//err := c.DB.SyncProject(syncInfo)
	//if err != nil {
	//	return echoutils.APIError(e, err)
	//}
	//
	//return echoutils.APIOk(e, nil)
}

func (c *Controller) projectDeps(e echo.Context) error {
	return echoutils.APIOk(e, nil)
	//TODO(buglloc): enable later
	//
	//if _, err := c.getUser(e); err != nil {
	//	return echoutils.APIError(e, err)
	//}
	//
	//service := e.Param("service")
	//name := e.Param("name")
	//deps, err := c.DB.LookupProjectDeps(service, name)
	//if err != nil {
	//	if err != db.ErrNotFound {
	//		return echoutils.APIError(e, err)
	//	}
	//	return echoutils.APINotFound(e)
	//}
	//
	//return echoutils.APIOk(e, deps)
}

func (c *Controller) getUser(e echo.Context) (*blackbox.User, error) {
	token := e.Request().Header.Get("Authorization")
	if len(token) <= 6 || token[:6] != "OAuth " {
		return nil, xerrors.New("failed to parse OAuth token")
	}

	rsp, err := c.BlackBox.OAuth(
		e.Request().Context(),
		blackbox.OAuthRequest{
			OAuthToken: token[6:],
			UserIP:     e.RealIP(),
		})

	if err != nil {
		return nil, xerrors.Errorf("failed to check OAuth token: %w", err)
	}

	return &rsp.User, nil
}
