package python

import (
	"fmt"
	"strconv"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/security/libs/go/pypi/pypipkg"
	"a.yandex-team.ru/security/yadi/web/internal/db"
	"a.yandex-team.ru/security/yadi/web/internal/echoutils"
	"a.yandex-team.ru/security/yadi/web/internal/infra"
	"a.yandex-team.ru/security/yadi/web/internal/models"
)

const (
	listLimit = 1000
)

type Controller struct {
	*infra.Infra
}

type listResult struct {
	Next     string                  `json:"next"`
	Packages []*models.PythonPackage `json:"packages"`
}

func (c *Controller) BuildRoute(g *echo.Group) error {
	g.GET("", c.pyListPackages)
	g.GET("/:pkg", c.pyPackage)
	g.GET("/:pkg/latest", c.pyPackageLatest)
	g.GET("/:pkg/:ver", c.pyPackageVer)
	return nil
}

func (c *Controller) pyListPackages(e echo.Context) error {
	var lastKey uint64
	if rawKey := e.QueryParam("key"); rawKey != "" {
		key, err := strconv.ParseUint(rawKey, 10, 64)
		if err != nil {
			return echoutils.APIError(e, err)
		}
		lastKey = key
	}
	lastName := e.QueryParam("name")

	pkgs, err := c.DB.ListPyPkgs(lastKey, lastName, listLimit)
	if err != nil {
		return echoutils.APIError(e, err)
	}

	result := listResult{
		Packages: pkgs,
	}

	if len(pkgs) > 0 {
		lastPkg := pkgs[len(pkgs)-1]
		result.Next = fmt.Sprintf(
			"%s/api/v1/python/?key=%d&name=%s",
			c.CFG.WebURL, lastPkg.ID, lastPkg.Name,
		)
	}

	return echoutils.APIOk(e, result)
}

func (c *Controller) pyPackage(e echo.Context) error {
	pkgName := pypipkg.NormalizeName(e.Param("pkg"))
	pkg, err := c.DB.LookupPyPkg(pkgName)
	if err != nil {
		if err == db.ErrNotFound {
			return echoutils.APINotFound(e)
		}
		return echoutils.APIError(e, err)
	}

	return echoutils.APIOk(e, pkg)
}

func (c *Controller) pyPackageLatest(e echo.Context) error {
	pkgName := pypipkg.NormalizeName(e.Param("pkg"))
	pkg, err := c.DB.LookupPyPkg(pkgName)
	if err != nil {
		if err == db.ErrNotFound {
			return echoutils.APINotFound(e)
		}
		return echoutils.APIError(e, err)
	}

	if len(pkg.Versions) == 0 {
		return echoutils.APIOk(e, nil)
	}

	// Indexer save sorted versions in reverse order
	pkgVer, err := c.DB.LookupPyPkgVersion(pkgName, pkg.Versions[0])
	if err != nil {
		if err == db.ErrNotFound {
			return echoutils.APINotFound(e)
		}
		return echoutils.APIError(e, err)
	}
	return echoutils.APIOk(e, pkgVer)
}

func (c *Controller) pyPackageVer(e echo.Context) error {
	pkgName := pypipkg.NormalizeName(e.Param("pkg"))
	pkgVer := e.Param("ver")
	pkg, err := c.DB.LookupPyPkgVersion(pkgName, pkgVer)
	if err != nil {
		if err == db.ErrNotFound {
			return echoutils.APINotFound(e)
		}
		return echoutils.APIError(e, err)
	}

	return echoutils.APIOk(e, pkg)
}
