package vulndb

import (
	"bytes"
	"html/template"
	"math"
	"net/http"
	"strconv"

	"github.com/labstack/echo/v4"
	"github.com/yuin/goldmark"
	"github.com/yuin/goldmark/extension"
	"github.com/yuin/goldmark/renderer"
	"github.com/yuin/goldmark/util"

	"a.yandex-team.ru/security/libs/go/goldmark/yrenderer"
	"a.yandex-team.ru/security/libs/go/simplelog"
	"a.yandex-team.ru/security/yadi/web/internal/echoutils"
	"a.yandex-team.ru/security/yadi/web/internal/infra"
	"a.yandex-team.ru/security/yadi/web/pkg/advisories"
)

const (
	vulnsPerPage = 200
)

var (
	md = goldmark.New(
		goldmark.WithExtensions(extension.GFM),
		goldmark.WithRendererOptions(
			renderer.WithNodeRenderers(
				util.Prioritized(yrenderer.NewHideReferer(), 0),
				util.Prioritized(yrenderer.NewFixedHeading(3), 0),
			),
		),
	)
)

type Controller struct {
	*infra.Infra
}

func (c *Controller) BuildRoute(g *echo.Group) error {
	g.GET("", c.list)
	g.GET("/:type", c.listByType)
	g.GET("/vuln/:id", c.detail)
	return nil
}

func (c *Controller) list(e echo.Context) error {
	advisories, err := c.VulnDB.AdvisoriesByType("all")
	if err != nil {
		return echoutils.PageError(e, err)
	}

	totalPages := int(math.Ceil(float64(len(advisories)) / float64(vulnsPerPage)))
	rawPage := e.QueryParam("page")
	curPage := 1
	if page, err := strconv.Atoi(rawPage); err == nil && page > 1 && page <= totalPages {
		curPage = page
	}

	leftBound := (curPage - 1) * vulnsPerPage
	rightBound := curPage * vulnsPerPage
	if rightBound > len(advisories) {
		rightBound = len(advisories)
	}

	return e.Render(http.StatusOK, "vulndb-list.html", echo.Map{
		"current":       "all",
		"advisories":    advisories[leftBound:rightBound],
		"advisoryTypes": c.VulnDB.AdvisoryTypes(),
		"curPage":       curPage,
		"totalPages":    totalPages,
		"stats":         c.VulnDB.Stats(),
	})
}

func (c *Controller) listByType(e echo.Context) error {
	listType := e.Param("type")
	advisories, err := c.VulnDB.AdvisoriesByType(listType)
	if err != nil {
		return echoutils.PageError(e, err)
	}

	totalPages := int(math.Ceil(float64(len(advisories)) / float64(vulnsPerPage)))
	rawPage := e.QueryParam("page")
	curPage := 1
	if page, err := strconv.Atoi(rawPage); err == nil && page > 1 && page <= totalPages {
		curPage = page
	}

	leftBound := (curPage - 1) * vulnsPerPage
	rightBound := curPage * vulnsPerPage
	if rightBound > len(advisories) {
		rightBound = len(advisories)
	}

	return e.Render(http.StatusOK, "vulndb-list.html", echo.Map{
		"current":       listType,
		"advisories":    advisories[leftBound:rightBound],
		"advisoryTypes": c.VulnDB.AdvisoryTypes(),
		"curPage":       curPage,
		"totalPages":    totalPages,
		"stats":         c.VulnDB.Stats(),
	})
}

func (c *Controller) detail(e echo.Context) error {
	advisory, err := c.VulnDB.AdvisoryByID(e.Param("id"))
	if err != nil {
		return echoutils.PageError(e, err)
	}

	var sourceURL advisories.ExternalReferences
	if len(advisory.ExternalReferences) > 0 {
		sourceURL = advisory.ExternalReferences[len(advisory.ExternalReferences)-1]
	}

	tmplParams := echo.Map{
		"advisory":        advisory,
		"sourceURL":       sourceURL,
		"richDescription": false,
		"description":     "",
	}

	if advisory.RichDescription {
		desc, err := renderMD(advisory.Desc)
		if err != nil {
			simplelog.Error("failed to render vuln description", "id", e.Param("id"), "err", err.Error())
		} else {
			tmplParams["richDescription"] = true
			tmplParams["description"] = template.HTML(desc)
		}
	}

	return e.Render(http.StatusOK, "vulndb-detail.html", tmplParams)
}

// TODO(buglloc): user identification?
//func (c *Controller) getUser(e echo.Context) (blackbox.User, error) {
//	sessID, err := e.Cookie("Session_id")
//	if err != nil {
//		return nil, xerrors.New("no session")
//	}
//
//	user, err := c.BlackBox.SessionIDWithContext(
//		e.Request().Context(),
//		blackbox.SessionIDRequest{
//			SessionID: sessID.Value,
//			UserIP:    e.RealIP(),
//			Host:      e.Request().Host,
//		})
//
//	if err != nil {
//		return nil, xerrors.Errorf("failed to check user session: %w", err)
//	}
//
//	return user, nil
//}

func renderMD(source string) (string, error) {
	var buf bytes.Buffer
	if err := md.Convert([]byte(source), &buf); err != nil {
		return "", err
	}
	return buf.String(), nil
}
