package db

import (
	"context"
	"encoding/json"
	"fmt"
	"strings"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/yadi/web/internal/models"
)

func (c *DB) LookupProjectInfo(service, name string) (result *models.Project, resultErr error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	resultErr = table.Retry(c.ctx, c.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, c.selectProjectQuery)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$service", ydb.UTF8Value(service)),
				table.ValueParam("$name", ydb.UTF8Value(name)),
			))
			return
		}),
	)

	if resultErr != nil {
		resultErr = xerrors.Errorf("failed to lookup yadi project: %w", resultErr)
		return
	}

	if !res.NextSet() || !res.NextRow() {
		resultErr = ErrNotFound
		return
	}

	result = &models.Project{
		Service: service,
		Name:    name,
	}

	// active, owners, repo_uri, severity, skips, targets

	res.SeekItem("active")
	result.Active = res.OBool()

	res.NextItem()
	result.Owners = res.OUTF8()

	res.NextItem()
	result.RepoURI = res.OUTF8()

	res.NextItem()
	result.Severity = res.OFloat()

	res.NextItem()
	skipsResult := res.OJSON()
	if len(skipsResult) > 0 {
		resultErr = json.Unmarshal([]byte(skipsResult), &result.Skips)
		if resultErr != nil {
			resultErr = xerrors.Errorf("failed to decode skips: %w", resultErr)
			return
		}
	}

	res.NextItem()
	targetsResult := res.OJSON()
	if len(targetsResult) > 0 {
		resultErr = json.Unmarshal([]byte(targetsResult), &result.Targets)
		if resultErr != nil {
			resultErr = xerrors.Errorf("failed to decode targets: %w", resultErr)
			return
		}
	}

	resultErr = res.Err()
	if resultErr != nil {
		resultErr = xerrors.Errorf("failed to lookup yadi project: %w", resultErr)
		return
	}

	return
}

func (c *DB) LookupProjectDeps(service, name string) (result *models.Dependencies, resultErr error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	resultErr = table.Retry(c.ctx, c.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, c.selectProjectDependenciesQuery)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$service", ydb.UTF8Value(service)),
				table.ValueParam("$name", ydb.UTF8Value(name)),
			))
			return
		}),
	)

	if resultErr != nil {
		resultErr = xerrors.Errorf("failed to lookup yadi project %s/%s dependencies: %w", service, name, resultErr)
		return
	}

	if !res.NextSet() || !res.NextRow() {
		resultErr = ErrNotFound
		return
	}

	result = new(models.Dependencies)

	// dependencies

	res.SeekItem("dependencies")
	resultDeps := res.OJSON()
	if len(resultDeps) > 0 {
		resultErr = json.Unmarshal([]byte(resultDeps), result)
		if resultErr != nil {
			resultErr = xerrors.Errorf("failed to decode dependencies: %w", resultErr)
			return
		}
	}

	resultErr = res.Err()
	if resultErr != nil {
		resultErr = xerrors.Errorf("failed to lookup yadi project %s/%s dependencies: %w", service, name, resultErr)
		return
	}

	return
}

func (c *DB) CreateProject(project *models.Project) (err error) {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	var targets string
	if len(project.Targets) > 0 {
		t, err := json.Marshal(project.Targets)
		if err != nil {
			return xerrors.Errorf("failed to encode skips: %w", err)
		}
		targets = string(t)
	} else {
		targets = "[]"
	}

	return table.Retry(c.ctx, c.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, c.createProjectQuery)
			if err != nil {
				return err
			}

			_, _, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$service", ydb.UTF8Value(project.Service)),
				table.ValueParam("$name", ydb.UTF8Value(project.Name)),
				table.ValueParam("$active", ydb.BoolValue(false)),
				table.ValueParam("$owners", ydb.UTF8Value(project.Owners)),
				table.ValueParam("$repoUri", ydb.UTF8Value(project.RepoURI)),
				table.ValueParam("$severity", ydb.FloatValue(project.Severity)),
				table.ValueParam("$targets", ydb.JSONValue(targets)),
				table.ValueParam("$createdAt", ydb.Int64Value(time.Now().Unix())),
			))
			return err
		}),
	)
}

func (c *DB) SyncProject(syncInfo *models.SyncInfo) (err error) {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	// TODO(buglloc): WTF?!
	params := table.NewQueryParameters(
		table.ValueParam("$service", ydb.UTF8Value(syncInfo.Service)),
		table.ValueParam("$name", ydb.UTF8Value(syncInfo.Name)),
		table.ValueParam("$updatedAt", ydb.Int64Value(time.Now().Unix())),
	)

	keys := []string{"id, service", "name", "updated_at"}
	values := []string{"Digest::CityHash($service||$name)", "$service", "$name", "$updatedAt"}
	query := strings.Builder{}
	query.WriteString(fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");
DECLARE $service AS Utf8;
DECLARE $name AS Utf8;
DECLARE $updatedAt AS Int64;`, c.webTablesPath))
	if syncInfo.Skips != nil {
		keys = append(keys, "skips")
		values = append(values, "$skips")
		query.WriteString(`DECLARE $skips AS Json;`)
		s, err := json.Marshal(syncInfo.Skips)
		if err != nil {
			return xerrors.Errorf("failed to encode skips: %w", err)
		}
		params.Add(table.ValueParam("$skips", ydb.JSONValue(string(s))))
	}

	if syncInfo.Targets != nil {
		keys = append(keys, "targets")
		values = append(values, "$targets")
		query.WriteString(`DECLARE $targets AS Json;`)
		t, err := json.Marshal(syncInfo.Targets)
		if err != nil {
			return xerrors.Errorf("failed to encode targets: %w", err)
		}
		params.Add(table.ValueParam("$targets", ydb.JSONValue(string(t))))
	}

	if syncInfo.Dependencies != nil {
		keys = append(keys, "dependencies")
		values = append(values, "$dependencies")
		query.WriteString(`DECLARE $dependencies AS Json;`)
		d, err := json.Marshal(syncInfo.Dependencies)
		if err != nil {
			return xerrors.Errorf("failed to encode dependencies: %w", err)
		}
		params.Add(table.ValueParam("$dependencies", ydb.JSONValue(string(d))))
	}

	if syncInfo.Severity != nil {
		keys = append(keys, "severity")
		values = append(values, "$severity")
		query.WriteString(`DECLARE $severity AS Float;`)
		params.Add(table.ValueParam("$severity", ydb.FloatValue(*syncInfo.Severity)))
	}

	query.WriteString(fmt.Sprintf(
		`UPSERT INTO projects (%s) VALUES (%s)`,
		strings.Join(keys, ","),
		strings.Join(values, ","),
	))

	return table.Retry(c.ctx, c.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, query.String())
			if err != nil {
				return err
			}

			_, _, err = stmt.Execute(ctx, writeTx, params)
			return err
		}),
	)
}

func (c *DB) ActivateProject(project *models.Project) (err error) {
	writeTx := table.TxControl(
		table.BeginTx(
			table.WithSerializableReadWrite(),
		),
		table.CommitTx(),
	)

	return table.Retry(c.ctx, c.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, c.activateProjectQuery)
			if err != nil {
				return err
			}

			_, _, err = stmt.Execute(ctx, writeTx, table.NewQueryParameters(
				table.ValueParam("$service", ydb.UTF8Value(project.Service)),
				table.ValueParam("$name", ydb.UTF8Value(project.Name)),
				table.ValueParam("$createdBy", ydb.Uint64Value(project.CreatedBy)),
				table.ValueParam("$updatedAt", ydb.Int64Value(time.Now().Unix())),
			))
			return err
		}),
	)
}
