package db

import (
	"context"
	"encoding/json"
	"strings"
	"time"

	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/security/yadi/web/internal/models"
)

func (c *DB) LookupPyPkgVersion(name, version string) (result *models.PythonPackageVersion, resultErr error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	resultErr = table.Retry(c.ctx, c.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, c.selectPyPkgVersionQuery)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$name", ydb.UTF8Value(name)),
				table.ValueParam("$version", ydb.UTF8Value(version)),
			))
			return
		}),
	)

	if resultErr != nil {
		resultErr = xerrors.Errorf("failed to lookup python pkg version: %w", resultErr)
		return
	}

	if !res.NextSet() || !res.NextRow() {
		resultErr = ErrNotFound
		return
	}

	result = &models.PythonPackageVersion{
		Name:    name,
		Version: version,
	}
	// license, pkg_url, requirements, updated_at

	res.SeekItem("license")
	result.License = res.OUTF8()

	res.NextItem()
	result.PkgURL = res.OUTF8()

	res.NextItem()
	requirements := res.OJSON()
	if len(requirements) > 0 {
		resultErr = json.Unmarshal([]byte(requirements), &result.Requirements)
		if resultErr != nil {
			resultErr = xerrors.Errorf("failed to decode requirements: %w", resultErr)
			return
		}
	}

	res.NextItem()
	result.UpdatedAt = time.Unix(res.OInt64(), 0)

	resultErr = res.Err()
	if resultErr != nil {
		resultErr = xerrors.Errorf("failed to lookup python pkg version: %w", resultErr)
		return
	}

	return
}

func (c *DB) LookupPyPkg(name string) (result *models.PythonPackage, resultErr error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	resultErr = table.Retry(c.ctx, c.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, c.selectPyPkgQuery)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$name", ydb.UTF8Value(name)),
			))
			return
		}),
	)

	if resultErr != nil {
		resultErr = xerrors.Errorf("failed to lookup python pkg: %w", resultErr)
		return
	}

	if !res.NextSet() || !res.NextRow() {
		resultErr = ErrNotFound
		return
	}

	result = &models.PythonPackage{
		Name: name,
	}

	// pkg_name, source, versions, updated_at

	res.SeekItem("pkg_name")
	result.PkgName = res.OUTF8()

	res.NextItem()
	result.Source = res.OUTF8()

	res.NextItem()
	if versions := res.OUTF8(); len(versions) > 0 {
		result.Versions = strings.Split(versions, VersionsDelimiter)
	}

	res.NextItem()
	result.UpdatedAt = time.Unix(res.OInt64(), 0)

	resultErr = res.Err()
	if resultErr != nil {
		resultErr = xerrors.Errorf("failed to lookup python pkg: %w", resultErr)
		return
	}
	return
}

func (c *DB) ListPyPkgs(lastKey uint64, lastName string, limit uint64) (result []*models.PythonPackage, resultErr error) {
	readTx := table.TxControl(
		table.BeginTx(
			table.WithOnlineReadOnly(),
		),
		table.CommitTx(),
	)

	var res *table.Result
	resultErr = table.Retry(c.ctx, c.sp,
		table.OperationFunc(func(ctx context.Context, s *table.Session) (err error) {
			stmt, err := s.Prepare(ctx, c.listPyPkgsQuery)
			if err != nil {
				return err
			}

			_, res, err = stmt.Execute(ctx, readTx, table.NewQueryParameters(
				table.ValueParam("$lastName", ydb.UTF8Value(lastName)),
				table.ValueParam("$lastKey", ydb.Uint64Value(lastKey)),
				table.ValueParam("$limit", ydb.Uint64Value(limit)),
			))
			return
		}),
	)

	if resultErr != nil {
		resultErr = xerrors.Errorf("failed to lookup python pkg: %w", resultErr)
		return
	}

	result = make([]*models.PythonPackage, 0, limit)
	for res.NextSet() {
		for res.NextRow() {
			res.NextItem()

			pkg := new(models.PythonPackage)
			// key, name, pkg_name, source, versions, updated_at

			res.SeekItem("key")
			pkg.ID = res.OUint64()

			res.NextItem()
			pkg.Name = res.OUTF8()

			res.NextItem()
			pkg.PkgName = res.OUTF8()

			res.NextItem()
			pkg.Source = res.OUTF8()

			res.NextItem()
			if versions := res.OUTF8(); len(versions) > 0 {
				pkg.Versions = strings.Split(versions, VersionsDelimiter)
			}

			res.NextItem()
			pkg.UpdatedAt = time.Unix(res.OInt64(), 0)

			result = append(result, pkg)
		}
	}

	resultErr = res.Err()
	if resultErr != nil {
		resultErr = xerrors.Errorf("failed to lookup python pkg: %w", resultErr)
		return
	}
	return
}
