package queries

import "fmt"

// Returns all "new" vulnerabilities
func SelectNewVulnsQuery(pathPrefix string) string {
	return fmt.Sprintf(`
/*
Запрос возвращает все новые уязвимости. Принимает ограничение по строкам в результате и последний ключ для сорировки.
Новые - значит те, которые еще не попали в таблицу [feed], т.е. не модерировались.
В список попадут те уязвимости, которые есть в [src] и [actions] по ключу srcId и srcType.
В список не попадут уязвимости с [src].isDeleted == True.
Здесь не проверяется [action].vulnAction.
*/

PRAGMA TablePathPrefix("%s");

DECLARE $lastKey AS Uint64;
DECLARE $limit AS Uint64;

-- "key" field in actions needed
$actions = (
  SELECT
    Digest::CityHash(a.srcType || a.srcId) as key,
    a.srcType as srcType,
    a.srcId as srcId,
    a.vulnAction as vulnAction
  FROM actions as a
  LEFT ONLY JOIN feed as f
    ON
      (f.key = Digest::CityHash(a.srcType || a.srcId)) AND
      (f.srcType = a.srcType) AND
      (f.srcId = a.srcId)
  WHERE
    Digest::CityHash(a.srcType || a.srcId) > $lastKey AND
    a.vulnAction != "delete"
  ORDER BY key
  LIMIT $limit
);

SELECT
    s.key AS key,
    a.srcType AS srcType,
    a.srcId AS srcId,
    a.vulnAction AS vulnAction,
    s.cvssScore AS cvssScore,
    s.vulnVersions AS vulnVersions,
    s.title AS title,
    s.pkgName AS pkgName,
    s.externalReferences AS externalReferences,
    s.description AS description,
    s.lang AS lang,
    s.patchedVersions AS patchedVersions,
    s.patchExists AS patchExists,
    s.updatedAt AS updatedAt,
    s.yaId AS yaId,
    s.disclosedAt AS disclosedAt,
    s.richDescription AS richDescription
FROM $actions AS a
INNER JOIN src as s USING(key, srcType, srcId)
WHERE
    s.isDeleted == False
;`, pathPrefix)
}

// Returns all "changed" vulnerabilities
func SelectChangedVulnsQuery(pathPrefix string) string {
	return fmt.Sprintf(`
-- changed vulns - vulns from "src" and "actions" that are exists in table "feed" (regardless of vulnAction)
/*
Запрос возвращает все измененные уязвимости. Принимает ограничение по строкам в результате и последний ключ для сорировки.
Измененные - значит те, которые есть И в таблице [src] И в таблице [feed] И появились в таблице [actions]
(незавимимо от значения [action].vulnAction.
JOIN идет по значеням ключей srcType и srcId.
*/

PRAGMA TablePathPrefix("%s");

DECLARE $lastKey AS Uint64;
DECLARE $limit AS Uint64;

-- "key" field in actions needed
$feed = (
  SELECT
    f.key as key,
    f.srcType as srcType,
    f.srcId as srcId,
    f.cvssScore as cvssScore,
    f.vulnVersions AS vulnVersions,
    f.title AS title,
    f.pkgName AS pkgName,
    f.lang AS lang,
    a.vulnAction as vulnAction
  FROM actions as a
  INNER JOIN feed as f ON
      (f.key = Digest::CityHash(a.srcType || a.srcId)) AND
      (f.srcType = a.srcType) AND
      (f.srcId = a.srcId)
  WHERE
    Digest::CityHash(a.srcType || a.srcId) > $lastKey
  ORDER BY key
  LIMIT $limit
);

$data = (
  SELECT
    f.key AS key,

    f.srcType AS srcType,
    f.srcId AS srcId,

    f.vulnAction AS vulnAction,

    f.cvssScore AS cvssScoreOld,
    s.cvssScore AS cvssScoreNew,

    f.vulnVersions AS vulnVersionsOld,
    s.vulnVersions AS vulnVersionsNew,

    f.title AS titleOld,
    s.title AS titleNew,

    f.pkgName AS pkgNameOld,
    s.pkgName AS pkgNameNew,

    f.lang AS langOld,
    s.lang AS langNew,

    s.externalReferences AS externalReferences,
    s.description AS description,
    s.patchedVersions AS patchedVersions,
    s.patchExists AS patchExists,
    s.updatedAt AS updatedAt,
    s.yaId AS yaId,
    s.disclosedAt AS disclosedAt,
    s.richDescription AS richDescription
  FROM $feed AS f
  INNER JOIN src as s USING(key, srcType, srcId)
);

SELECT
    key, srcType, srcId, vulnAction,
    cvssScoreOld AS cvssScore,
    vulnVersionsOld AS vulnVersions,
    titleOld AS title,
    pkgNameOld AS pkgName,
    langOld AS lang,
    externalReferences,
    description,
    patchedVersions,
    patchExists,
    updatedAt,
    yaId,
    disclosedAt,
    richDescription,
    "old" as generation
FROM $data
;

SELECT
    key, srcType, srcId, vulnAction,
    cvssScoreNew AS cvssScore,
    vulnVersionsNew AS vulnVersions,
    titleNew AS title,
    pkgNameNew AS pkgName,
    langNew AS lang,
    externalReferences,
    description,
    patchedVersions,
    patchExists,
    updatedAt,
    yaId,
    disclosedAt,
    richDescription,
    "new" as generation
FROM $data
;`, pathPrefix)
}

// Returns all vulnerabilities that are already moderated
func SelectFullFeedQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $lastKey AS Uint64;
DECLARE $limit AS Uint64;

-- уязвимсоти из фида _всегда_ должны быть в src, поэтому проще сначала выбрать записи из фида, а потом уже поджойнить
$feed = (
  SELECT *
  FROM feed
  WHERE key > $lastKey
  LIMIT $limit
);

SELECT
    f.key AS key,
    f.srcType AS srcType,
    f.srcId AS srcId,

    f.pkgName AS pkgName,
    f.lang AS lang,
    f.cvssScore AS cvssScore,
    f.vulnVersions AS vulnVersions,
    f.title AS title,

    s.patchedVersions AS patchedVersions,
    s.description AS description,
    s.patchExists AS patchExists,
    s.externalReferences AS externalReferences,
    s.disclosedAt AS disclosedAt,
    s.richDescription AS richDescription,
    s.yaId AS yaId
FROM $feed AS f
INNER JOIN src as s USING(key, srcType, srcId)
;`, pathPrefix)
}

// Returns one vulnerability and related action from [src] table
func SelectOneSrcVulnWithActionQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $srcType AS Utf8;
DECLARE $srcId AS Utf8;

SELECT
    s.key AS key,
    s.srcType AS srcType,
    s.srcId AS srcId,
    s.pkgName AS pkgName,
    s.lang AS lang,
    s.cvssScore AS cvssScore,
    s.vulnVersions AS vulnVersions,
    s.title AS title,
    s.patchedVersions AS patchedVersions,
    s.description AS description,
    s.patchExists AS patchExists,
    s.externalReferences AS externalReferences,
    s.disclosedAt AS disclosedAt,
    s.richDescription AS richDescription,
    s.yaId AS yaId,
    a.vulnAction as vulnAction
FROM
src as s
JOIN
    actions as a
USING (srcType, srcId)
WHERE
    s.key = Digest::CityHash($srcType || $srcId) AND
    s.srcType = $srcType AND
    s.srcId = $srcId AND
    s.isDeleted == False
LIMIT 1
;`, pathPrefix)
}

// Returns one vulnerability that is already moderated specified by yaId
func SelectOneFeedVulnQueryByYaID(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $yaId AS Utf8;

SELECT
    s.srcType AS srcType,
    s.srcId AS srcId,

    f.pkgName AS pkgName,
    f.lang AS lang,
    f.cvssScore AS cvssScore,
    f.vulnVersions AS vulnVersions,
    f.title AS title,

    s.patchedVersions AS patchedVersions,
    s.description AS description,
    s.patchExists AS patchExists,
    s.externalReferences AS externalReferences,
    s.disclosedAt AS disclosedAt,
    s.richDescription AS richDescription,
    s.yaId AS yaId
FROM src AS s
INNER JOIN feed as f ON USING(key, srcType, srcId)
WHERE
    s.yaId = $yaId AND
    s.isDeleted == False
;`, pathPrefix)
}

// Returns one vulnerability that is already moderated specified by srcType and srcId
func SelectOneFeedVulnQueryBySrc(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $srcType AS Utf8;
DECLARE $srcId AS Utf8;

SELECT
    f.srcType AS srcType,
    f.srcId AS srcId,

    f.pkgName AS pkgName,
    f.lang AS lang,
    f.cvssScore AS cvssScore,
    f.vulnVersions AS vulnVersions,
    f.title AS title,

    s.patchedVersions AS patchedVersions,
    s.description AS description,
    s.patchExists AS patchExists,
    s.externalReferences AS externalReferences,
    s.disclosedAt AS disclosedAt,
    s.richDescription AS richDescription,
    s.yaId AS yaId
FROM feed as f
INNER JOIN src AS s USING(key, srcType, srcId)
WHERE
    f.key = Digest::CityHash($srcType || $srcId) AND
    f.srcType = $srcType AND
    f.srcId = $srcId AND
    s.isDeleted == False
;`, pathPrefix)
}

// Delete action for one specific vulnerability
func DeleteActionQuery(pathPrefix string) string {
	return fmt.Sprintf(`
PRAGMA TablePathPrefix("%s");

DECLARE $srcType AS Utf8;
DECLARE $srcId AS Utf8;

DELETE FROM actions
WHERE
    srcType = $srcType AND
    srcId = $srcId
;`, pathPrefix)
}

// Change vulnerability in the [feed] table
func ChangeVulnQuery(pathPrefix string) string {
	return fmt.Sprintf(`
/*
Запрос изменяет все колонки уязвимости в таблице [feed] по составному ключу (srcType, srcId).
После UPSERT в [feed] уязвимость с таким же ключом (srcType, srcId) удаляется из [actions].
*/

PRAGMA TablePathPrefix("%s");

DECLARE $vuln AS "List<Struct<
    srcType: Utf8,
    srcId: Utf8,
    cvssScore: Float,
    vulnVersions: Utf8,
    title: Utf8,
    pkgName: Utf8,
    lang: Utf8>>";

UPSERT INTO feed
    (key, srcType, srcId, cvssScore, vulnVersions, title, pkgName, lang)
SELECT
    Digest::CityHash(srcType || srcId),
    srcType,
    srcId,
    cvssScore,
    vulnVersions,
    title,
    pkgName,
    lang
FROM AS_TABLE($vuln)
;

DELETE FROM actions
WHERE AsTuple(srcType, srcId) IN (
    SELECT AsTuple(srcType, srcId) FROM AS_TABLE($vuln)
);`, pathPrefix)
}
